<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/api/esim-api.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: /login.php');
    exit;
}

// Get eSIM ID
$esimId = isset($_GET['id']) ? intval($_GET['id']) : 0;

if (!$esimId) {
    header('Location: /my-esims.php');
    exit;
}

// Get eSIM from database
$conn = getDBConnection();
$stmt = $conn->prepare("SELECT * FROM orders WHERE id = ? AND user_id = ?");
$stmt->bind_param("ii", $esimId, $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
$esim = $result->fetch_assoc();
$stmt->close();
$conn->close();

if (!$esim) {
    header('Location: /my-esims.php');
    exit;
}

// Initialize API
$api = new eSIMAPI();

// Initialize variables
$apiData = null;
$isExpired = false;
$isActive = false;
$totalData = null;
$usedData = null;
$remainingData = null;
$expiryDate = null;
$activationTime = null;
$network = null;

// Try to get API data
if (!empty($esim['iccid'])) {
    try {
        $apiResponse = $api->queryProfile($esim['iccid']);
        
        $isSuccess = false;
        if (isset($apiResponse['code']) && $apiResponse['code'] == 0) {
            $isSuccess = true;
        } elseif (isset($apiResponse['success']) && $apiResponse['success'] === true) {
            $isSuccess = true;
        } elseif (isset($apiResponse['errorCode']) && $apiResponse['errorCode'] === '0') {
            $isSuccess = true;
        } elseif ($apiResponse && isset($apiResponse['obj'])) {
            // اگه obj موجود باشه، احتمالاً موفقه
            $isSuccess = true;
        }
        
        if ($apiResponse && $isSuccess) {
            // روش‌های مختلف پیدا کردن esimList
            $esimList = null;
            if (isset($apiResponse['obj']['esimList'])) {
                $esimList = $apiResponse['obj']['esimList'];
            } elseif (isset($apiResponse['obj']) && is_array($apiResponse['obj'])) {
                $esimList = $apiResponse['obj'];
            } elseif (isset($apiResponse['data']['esimList'])) {
                $esimList = $apiResponse['data']['esimList'];
            } elseif (isset($apiResponse['esimList'])) {
                $esimList = $apiResponse['esimList'];
            }
            
            if (is_array($esimList) && !empty($esimList)) {
                // اگه آرایه‌ای از eSIM هاست، اولی رو بگیر
                $apiData = isset($esimList[0]) && is_array($esimList[0]) ? $esimList[0] : $esimList;
                
                // وضعیت: esimStatus - وضعیت‌های فعال: ACTIVE, IN_USE, ACTIVATED, INSTALLED
                $currentStatus = $apiData['esimStatus'] ?? ($apiData['status'] ?? '');
                $activeStatuses = ['ACTIVE', 'IN_USE', 'ACTIVATED', 'INSTALLED'];
                $isActive = in_array(strtoupper($currentStatus), $activeStatuses);
                
                // تاریخ انقضا: expiredTime یا expiryTime یا expireTime
                $expiryField = $apiData['expiredTime'] ?? ($apiData['expiryTime'] ?? ($apiData['expireTime'] ?? null));
                if ($expiryField) {
                    $expiryDate = $expiryField;
                    $expiryTimestamp = strtotime($expiryDate);
                    $isExpired = $expiryTimestamp < time();
                }
                
                // زمان فعال‌سازی: activateTime یا activationTime
                $activateField = $apiData['activateTime'] ?? ($apiData['activationTime'] ?? null);
                if ($activateField) {
                    $activationTime = $activateField;
                }
                
                // APN یا شبکه
                $network = $apiData['apn'] ?? ($apiData['network'] ?? ($apiData['operator'] ?? null));
                
                // حجم داده: چندین نام مختلف رو چک کن
                $totalField = $apiData['totalVolume'] ?? ($apiData['totalData'] ?? ($apiData['dataVolume'] ?? ($apiData['volume'] ?? null)));
                $usedField = $apiData['orderUsage'] ?? ($apiData['usedVolume'] ?? ($apiData['usedData'] ?? ($apiData['usage'] ?? 0)));
                
                if ($totalField !== null) {
                    $totalData = (int)$totalField;
                    $usedData = (int)$usedField;
                    $remainingData = $totalData - $usedData;
                }
            }
        }
    } catch (Exception $e) {
        // Log error for debugging
        error_log("eSIM Details API Error: " . $e->getMessage());
    }
}

// Fallback to local data
if (!$apiData && isset($esim['validity_days']) && isset($esim['created_at'])) {
    $createdTime = strtotime($esim['created_at']);
    $expiryTime = $createdTime + ($esim['validity_days'] * 24 * 60 * 60);
    $isExpired = $expiryTime < time();
    $expiryDate = date('Y-m-d H:i:s', $expiryTime);
}

$usagePercentage = 0;
if ($totalData > 0 && $usedData !== null) {
    $usagePercentage = round(($usedData / $totalData) * 100, 1);
}

// تعیین وضعیت نهایی بر اساس داده‌های API و installationTime
$currentStatus = $apiData['esimStatus'] ?? ($apiData['status'] ?? '');
$hasInstallationTime = !empty($apiData['installationTime']);

if ($isExpired) {
    $statusText = 'منقضی';
    $statusIcon = 'fa-times-circle';
    $statusClass = 'status-expired';
} elseif (in_array(strtoupper($currentStatus), ['ACTIVE', 'IN_USE', 'ACTIVATED'])) {
    $statusText = 'فعال';
    $statusIcon = 'fa-check-circle';
    $statusClass = 'status-active';
} elseif (in_array(strtoupper($currentStatus), ['INSTALLED']) || $hasInstallationTime) {
    // اگر installationTime پر شده، یعنی نصب شده (حتی اگه esimStatus هنوز GOT_RESOURCE باشه)
    $statusText = 'نصب شده';
    $statusIcon = 'fa-mobile-alt';
    $statusClass = 'status-active';
} elseif (in_array(strtoupper($currentStatus), ['GOT_RESOURCE', 'RELEASED'])) {
    $statusText = 'آماده نصب';
    $statusIcon = 'fa-download';
    $statusClass = 'status-active';
} elseif (in_array(strtoupper($currentStatus), ['DEPLETED'])) {
    $statusText = 'حجم تمام شده';
    $statusIcon = 'fa-exclamation-triangle';
    $statusClass = 'status-expired';
} else {
    $statusText = $isActive ? 'فعال' : 'آماده نصب';
    $statusIcon = $isActive ? 'fa-check-circle' : 'fa-download';
    $statusClass = 'status-active';
}

// Days remaining
$daysRemaining = 0;
if ($expiryDate) {
    $expiryTimestamp = strtotime($expiryDate);
    $daysRemaining = max(0, floor(($expiryTimestamp - time()) / (60 * 60 * 24)));
} elseif (isset($esim['validity_days']) && isset($esim['created_at'])) {
    $createdTime = strtotime($esim['created_at']);
    $expiryTime = $createdTime + ($esim['validity_days'] * 24 * 60 * 60);
    $daysRemaining = max(0, floor(($expiryTime - time()) / (60 * 60 * 24)));
}

include __DIR__ . '/includes/header.php';
?>

<link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;600;700;800&display=swap" rel="stylesheet">

<style>
* {
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif;
}

body {
    background: #f5f5f5 !important;
    min-height: 100vh;
    padding: 80px 20px 20px;
    direction: rtl;
}

.back-button {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    background: white;
    color: #6b7ff5;
    padding: 10px 20px;
    border-radius: 10px;
    text-decoration: none;
    font-weight: 600;
    margin-bottom: 20px;
    transition: all 0.3s ease;
}

.back-button:hover {
    background: #6b7ff5;
    color: white;
    transform: translateX(-5px);
}

.details-container {
    max-width: 900px;
    margin: 0 auto;
}

.esim-header {
    background: white;
    border-radius: 20px;
    padding: 30px;
    margin-bottom: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    text-align: center;
}

.header-flag {
    width: 120px;
    height: 85px;
    margin: 0 auto 20px;
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
}

.header-flag img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.header-country {
    font-size: 2rem;
    font-weight: 800;
    color: #1e293b;
    margin-bottom: 10px;
}

.header-package {
    font-size: 1.2rem;
    color: #64748b;
    font-weight: 600;
    margin-bottom: 15px;
}

.status-badge-large {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    border-radius: 25px;
    font-weight: 700;
    font-size: 1rem;
}

.status-active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.status-expired {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.details-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 20px;
    margin-bottom: 20px;
}

.detail-card {
    background: white;
    border-radius: 15px;
    padding: 25px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.15);
}

.detail-card-title {
    display: flex;
    align-items: center;
    gap: 10px;
    font-size: 1.1rem;
    font-weight: 700;
    color: #475569;
    margin-bottom: 20px;
}

.detail-card-title i {
    color: #6b7ff5;
    font-size: 1.3rem;
}

.full-width {
    grid-column: 1 / -1;
}

/* Circular Progress */
.circular-progress-wrapper {
    display: flex;
    align-items: center;
    justify-content: space-around;
    gap: 30px;
}

.circular-progress {
    position: relative;
    width: 180px;
    height: 180px;
    flex-shrink: 0;
}

.circular-progress-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    text-align: center;
}

.circular-percentage {
    font-size: 2.5rem;
    font-weight: 800;
    color: #6b7ff5;
    line-height: 1;
    margin-bottom: 5px;
}

.circular-label {
    font-size: 0.8rem;
    color: #64748b;
    font-weight: 600;
}

.circular-stats {
    display: flex;
    flex-direction: column;
    gap: 15px;
    flex: 1;
}

.circular-stat-item {
    display: flex;
    align-items: center;
    gap: 12px;
    background: #f8fafc;
    padding: 12px;
    border-radius: 12px;
}

.circular-stat-icon {
    width: 45px;
    height: 45px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    flex-shrink: 0;
}

.circular-stat-content {
    flex: 1;
}

.circular-stat-label {
    font-size: 0.75rem;
    color: #64748b;
    font-weight: 600;
    margin-bottom: 3px;
}

.circular-stat-value {
    font-size: 1rem;
    color: #1e293b;
    font-weight: 700;
}

.not-activated-message {
    display: flex;
    align-items: center;
    gap: 15px;
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    padding: 20px;
    border-radius: 12px;
    color: white;
}

.not-activated-icon {
    width: 50px;
    height: 50px;
    background: rgba(255,255,255,0.25);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    flex-shrink: 0;
}

.not-activated-title {
    font-size: 1rem;
    font-weight: 700;
    margin-bottom: 5px;
}

.not-activated-desc {
    font-size: 0.85rem;
    opacity: 0.95;
}

/* Time Box */
.time-box {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border-radius: 15px;
    padding: 25px;
    display: flex;
    align-items: center;
    gap: 20px;
    color: white;
}

.time-icon-large {
    width: 70px;
    height: 70px;
    background: rgba(255,255,255,0.25);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    flex-shrink: 0;
}

.time-content {
    flex: 1;
}

.time-label {
    font-size: 0.9rem;
    opacity: 0.9;
    margin-bottom: 5px;
}

.time-value {
    font-size: 2.5rem;
    font-weight: 800;
}

/* Info List */
.info-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.info-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid #e2e8f0;
}

.info-row:last-child {
    border-bottom: none;
}

.info-label {
    color: #64748b;
    font-weight: 600;
    font-size: 0.9rem;
}

.info-value {
    color: #1e293b;
    font-weight: 700;
    font-size: 0.95rem;
}

.status-active-text {
    color: #10b981;
}

.status-inactive-text {
    color: #ef4444;
}

/* QR Code */
.qr-wrapper {
    text-align: center;
    margin-bottom: 20px;
}

.qr-wrapper img {
    max-width: 250px;
    border-radius: 15px;
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

.btn-auto-add {
    display: inline-flex;
    align-items: center;
    gap: 10px;
    background: linear-gradient(135deg, #6b7ff5 0%, #8a96f7 100%);
    color: white;
    padding: 12px 25px;
    border-radius: 12px;
    text-decoration: none;
    font-weight: 700;
    margin-top: 15px;
    transition: all 0.3s ease;
}

.btn-auto-add:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(107, 127, 245, 0.4);
}

/* Activation Code */
.activation-code-box {
    background: #f8fafc;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    padding: 15px;
    margin-bottom: 15px;
}

.activation-label {
    color: #64748b;
    font-weight: 600;
    font-size: 0.85rem;
    margin-bottom: 8px;
    display: block;
}

.activation-code {
    font-family: 'Courier New', monospace;
    background: white;
    padding: 12px;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    font-size: 0.85rem;
    color: #1e293b;
    word-break: break-all;
    margin-bottom: 10px;
}

.btn-copy {
    background: #6b7ff5;
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 8px;
    width: 100%;
    justify-content: center;
    transition: all 0.3s ease;
}

.btn-copy:hover {
    background: #5a6de8;
}

/* Instructions */
.instructions-content {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.instruction-section {
    background: #f8fafc;
    padding: 20px;
    border-radius: 12px;
    border: 1px solid #e2e8f0;
}

.instruction-section h4 {
    color: #1e293b;
    font-size: 1.1rem;
    font-weight: 700;
    margin-bottom: 15px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.instruction-section h4 i {
    color: #6b7ff5;
    font-size: 1.3rem;
}

.instruction-section ol {
    margin: 0;
    padding-right: 20px;
    color: #475569;
}

.instruction-section li {
    margin-bottom: 10px;
    line-height: 1.6;
}

.instruction-section li:last-child {
    margin-bottom: 0;
}

.note-box {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border: 2px solid #fbbf24;
    border-radius: 12px;
    padding: 15px;
    display: flex;
    align-items: flex-start;
    gap: 12px;
}

.note-box i {
    color: #d97706;
    font-size: 1.3rem;
    flex-shrink: 0;
    margin-top: 2px;
}

.note-box strong {
    color: #92400e;
}

/* Tablet */
@media (max-width: 992px) {
    body {
        padding: 75px 15px 20px;
    }
    
    .details-container {
        max-width: 100%;
    }
    
    .details-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
}

/* Mobile */
@media (max-width: 768px) {
    body {
        padding: 70px 12px 20px;
    }
    
    .back-button {
        padding: 10px 18px;
        font-size: 0.9rem;
        margin-bottom: 15px;
    }
    
    .details-container {
        max-width: 100%;
    }
    
    .esim-header {
        padding: 25px 20px;
        border-radius: 16px;
        margin-bottom: 15px;
    }
    
    .header-flag {
        width: 100px;
        height: 70px;
        margin-bottom: 15px;
    }
    
    .header-country {
        font-size: 1.5rem;
        margin-bottom: 8px;
    }
    
    .header-package {
        font-size: 0.95rem;
    }
    
    .status-badge {
        padding: 8px 18px;
        font-size: 0.85rem;
    }
    
    .details-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .detail-card {
        padding: 20px 18px;
        border-radius: 14px;
    }
    
    .card-header h3 {
        font-size: 1.1rem;
    }
    
    .circular-progress-wrapper {
        flex-direction: column;
        gap: 25px;
    }
    
    .progress-item {
        flex: 1;
        max-width: 100%;
    }
    
    .progress-bar {
        width: 180px;
        height: 180px;
    }
    
    .progress-text {
        font-size: 1.8rem;
    }
    
    .time-box {
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }
    
    .time-icon {
        font-size: 2rem;
        margin: 0;
    }
    
    .time-value {
        font-size: 2rem;
    }
    
    .info-row {
        padding: 12px 0;
        font-size: 0.9rem;
    }
    
    .info-label {
        font-size: 0.85rem;
    }
    
    .info-value {
        font-size: 0.9rem;
    }
    
    .qr-wrapper img {
        max-width: 200px;
    }
    
    .btn-auto-add {
        padding: 12px 20px;
        font-size: 0.9rem;
    }
    
    .activation-code-box {
        padding: 15px;
    }
    
    .activation-code {
        font-size: 0.8rem;
        padding: 10px;
    }
    
    .btn-copy {
        padding: 10px 18px;
        font-size: 0.9rem;
    }
    
    .instruction-section {
        padding: 18px;
    }
    
    .instruction-section h4 {
        font-size: 1rem;
    }
    
    .instruction-section p, 
    .instruction-section ol {
        font-size: 0.9rem;
    }
}

/* Small Mobile */
@media (max-width: 480px) {
    body {
        padding: 70px 10px 20px;
    }
    
    .back-button {
        padding: 9px 16px;
        font-size: 0.85rem;
    }
    
    .esim-header {
        padding: 20px 16px;
        border-radius: 14px;
    }
    
    .header-flag {
        width: 85px;
        height: 60px;
    }
    
    .header-country {
        font-size: 1.3rem;
    }
    
    .header-package {
        font-size: 0.85rem;
    }
    
    .status-badge {
        padding: 7px 16px;
        font-size: 0.8rem;
    }
    
    .detail-card {
        padding: 18px 15px;
        border-radius: 12px;
    }
    
    .card-header h3 {
        font-size: 1rem;
    }
    
    .progress-bar {
        width: 160px;
        height: 160px;
    }
    
    .progress-text {
        font-size: 1.6rem;
    }
    
    .progress-label {
        font-size: 0.85rem;
    }
    
    .time-value {
        font-size: 1.8rem;
    }
    
    .time-label {
        font-size: 0.85rem;
    }
    
    .info-row {
        padding: 10px 0;
        font-size: 0.85rem;
    }
    
    .info-label {
        font-size: 0.8rem;
    }
    
    .info-value {
        font-size: 0.85rem;
    }
    
    .qr-wrapper img {
        max-width: 180px;
    }
    
    .btn-auto-add {
        padding: 11px 18px;
        font-size: 0.85rem;
    }
    
    .activation-code-box {
        padding: 12px;
    }
    
    .activation-code {
        font-size: 0.75rem;
        padding: 10px;
    }
    
    .btn-copy {
        padding: 9px 16px;
        font-size: 0.85rem;
    }
    
    .instruction-section {
        padding: 16px;
    }
    
    .instruction-section h4 {
        font-size: 0.95rem;
    }
    
    .instruction-section p, 
    .instruction-section ol {
        font-size: 0.85rem;
    }
    
    .instruction-section ol li {
        padding: 8px 0;
    }
}

/* Extra Small Mobile */
@media (max-width: 360px) {
    body {
        padding: 70px 8px 20px;
    }
    
    .back-button {
        padding: 8px 14px;
        font-size: 0.8rem;
    }
    
    .esim-header {
        padding: 18px 14px;
    }
    
    .header-flag {
        width: 75px;
        height: 55px;
    }
    
    .header-country {
        font-size: 1.2rem;
    }
    
    .header-package {
        font-size: 0.8rem;
    }
    
    .detail-card {
        padding: 16px 12px;
    }
    
    .card-header h3 {
        font-size: 0.95rem;
    }
    
    .progress-bar {
        width: 140px;
        height: 140px;
    }
    
    .progress-text {
        font-size: 1.4rem;
    }
    
    .time-value {
        font-size: 1.6rem;
    }
    
    .qr-wrapper img {
        max-width: 160px;
    }
}
</style>

<div class="details-container">
    <a href="/my-esims.php" class="back-button">
        <i class="fas fa-arrow-right"></i>
        <span>بازگشت به لیست</span>
    </a>
    
    <!-- Header -->
    <div class="esim-header">
        <div class="header-flag">
            <?php if (!empty($esim['country_code'])): ?>
            <img src="https://flagcdn.com/w320/<?php echo strtolower($esim['country_code']); ?>.png" 
                 alt="<?php echo htmlspecialchars($esim['country_name'] ?? ''); ?>">
            <?php endif; ?>
        </div>
        
        <div class="header-country"><?php echo htmlspecialchars($esim['country_name'] ?? ''); ?></div>
        <div class="header-package"><?php echo htmlspecialchars($esim['package_name'] ?? 'eSIM'); ?></div>
        
        <div class="status-badge-large <?php echo $statusClass; ?>">
            <i class="fas <?php echo $statusIcon; ?>"></i>
            <span><?php echo $statusText; ?></span>
        </div>
    </div>
    
    <!-- پیام وضعیت بر اساس esimStatus و installationTime -->
    <?php 
    $currentApiStatus = $apiData['esimStatus'] ?? '';
    $installationTime = $apiData['installationTime'] ?? null;
    
    // اگر installationTime پر شده، یعنی نصب شده (حتی اگه esimStatus هنوز GOT_RESOURCE باشه)
    $isInstalled = !empty($installationTime);
    ?>
    
    <?php if (in_array($currentApiStatus, ['GOT_RESOURCE', 'RELEASED']) && !$isInstalled): ?>
    <!-- فقط آماده نصب - هنوز نصب نشده -->
    <div style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 15px; padding: 20px; margin-bottom: 20px; color: white; display: flex; align-items: center; gap: 20px; box-shadow: 0 5px 20px rgba(245, 158, 11, 0.3);">
        <div style="width: 60px; height: 60px; background: rgba(255,255,255,0.25); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.8rem; flex-shrink: 0;">
            <i class="fas fa-download"></i>
        </div>
        <div style="flex: 1;">
            <div style="font-size: 1.2rem; font-weight: 800; margin-bottom: 8px;">
                ⏳ eSIM آماده نصب است
            </div>
            <div style="font-size: 0.95rem; opacity: 0.95; line-height: 1.6;">
                لطفاً QR Code را اسکن کنید و eSIM را روی گوشی نصب کنید.
            </div>
        </div>
    </div>
    <?php elseif ($isInstalled && !in_array($currentApiStatus, ['ACTIVE', 'IN_USE', 'ACTIVATED'])): ?>
    <!-- نصب شده اما هنوز فعال نشده -->
    <div style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); border-radius: 15px; padding: 20px; margin-bottom: 20px; color: white; display: flex; align-items: center; gap: 20px; box-shadow: 0 5px 20px rgba(59, 130, 246, 0.3);">
        <div style="width: 60px; height: 60px; background: rgba(255,255,255,0.25); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.8rem; flex-shrink: 0;">
            <i class="fas fa-mobile-alt"></i>
        </div>
        <div style="flex: 1;">
            <div style="font-size: 1.2rem; font-weight: 800; margin-bottom: 8px;">
                📲 eSIM نصب شده است
            </div>
            <div style="font-size: 0.95rem; opacity: 0.95; line-height: 1.6;">
                eSIM در تاریخ <strong><?php echo date('Y/m/d H:i', strtotime($installationTime)); ?></strong> روی دستگاه نصب شده است.
                <br>
                <strong>پس از اولین اتصال به شبکه موبایل در کشور مقصد</strong>، وضعیت به «فعال» تغییر می‌کند و اطلاعات مصرف نمایش داده می‌شود.
            </div>
        </div>
    </div>
    <?php elseif ($currentApiStatus == 'INSTALLED'): ?>
    <div style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); border-radius: 15px; padding: 20px; margin-bottom: 20px; color: white; display: flex; align-items: center; gap: 20px; box-shadow: 0 5px 20px rgba(59, 130, 246, 0.3);">
        <div style="width: 60px; height: 60px; background: rgba(255,255,255,0.25); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.8rem; flex-shrink: 0;">
            <i class="fas fa-mobile-alt"></i>
        </div>
        <div style="flex: 1;">
            <div style="font-size: 1.2rem; font-weight: 800; margin-bottom: 8px;">
                📲 eSIM نصب شده است
            </div>
            <div style="font-size: 0.95rem; opacity: 0.95; line-height: 1.6;">
                eSIM با موفقیت روی دستگاه شما نصب شده.
                <br>
                <strong>پس از اتصال به شبکه موبایل</strong>، اطلاعات مصرف به‌روزرسانی خواهد شد.
            </div>
        </div>
    </div>
    <?php elseif (in_array($currentApiStatus, ['ACTIVE', 'IN_USE', 'ACTIVATED'])): ?>
    <div style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); border-radius: 15px; padding: 20px; margin-bottom: 20px; color: white; display: flex; align-items: center; gap: 20px; box-shadow: 0 5px 20px rgba(16, 185, 129, 0.3);">
        <div style="width: 60px; height: 60px; background: rgba(255,255,255,0.25); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.8rem; flex-shrink: 0;">
            <i class="fas fa-check-circle"></i>
        </div>
        <div style="flex: 1;">
            <div style="font-size: 1.2rem; font-weight: 800; margin-bottom: 8px;">
                🟢 eSIM فعال است
            </div>
            <div style="font-size: 0.95rem; opacity: 0.95; line-height: 1.6;">
                eSIM شما فعال و در حال استفاده است.
                <?php if ($activationTime): ?>
                <br>زمان فعال‌سازی: <?php echo date('Y/m/d H:i', strtotime($activationTime)); ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php elseif ($currentApiStatus == 'DEPLETED'): ?>
    <div style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); border-radius: 15px; padding: 20px; margin-bottom: 20px; color: white; display: flex; align-items: center; gap: 20px; box-shadow: 0 5px 20px rgba(239, 68, 68, 0.3);">
        <div style="width: 60px; height: 60px; background: rgba(255,255,255,0.25); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.8rem; flex-shrink: 0;">
            <i class="fas fa-exclamation-triangle"></i>
        </div>
        <div style="flex: 1;">
            <div style="font-size: 1.2rem; font-weight: 800; margin-bottom: 8px;">
                ⚠️ حجم داده تمام شده
            </div>
            <div style="font-size: 0.95rem; opacity: 0.95; line-height: 1.6;">
                حجم داده این eSIM به پایان رسیده است. می‌توانید با خرید بسته شارژ، حجم جدید اضافه کنید.
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Details Grid -->
    <div class="details-grid">
        <!-- Data Usage -->
        <div class="detail-card full-width">
            <div class="detail-card-title">
                <i class="fas fa-chart-pie"></i>
                <span>مصرف حجم داده</span>
            </div>
            
            <?php if ($totalData !== null && $usedData !== null): ?>
            <div class="circular-progress-wrapper">
                <div class="circular-progress">
                    <svg width="180" height="180" viewBox="0 0 180 180">
                        <circle cx="90" cy="90" r="75" fill="none" stroke="#e2e8f0" stroke-width="15"/>
                        <circle cx="90" cy="90" r="75" fill="none" 
                                stroke="url(#gradient)" 
                                stroke-width="15"
                                stroke-dasharray="<?php echo ($usagePercentage * 471.24 / 100); ?> 471.24"
                                stroke-linecap="round"
                                transform="rotate(-90 90 90)"/>
                        <defs>
                            <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                                <stop offset="0%" style="stop-color:#6b7ff5;stop-opacity:1" />
                                <stop offset="100%" style="stop-color:#8a96f7;stop-opacity:1" />
                            </linearGradient>
                        </defs>
                    </svg>
                    
                    <div class="circular-progress-text">
                        <div class="circular-percentage"><?php echo $usagePercentage; ?>%</div>
                        <div class="circular-label">مصرف شده</div>
                    </div>
                </div>
                
                <div class="circular-stats">
                    <div class="circular-stat-item">
                        <div class="circular-stat-icon" style="background: linear-gradient(135deg, #6b7ff5 0%, #8a96f7 100%);">
                            <i class="fas fa-arrow-up"></i>
                        </div>
                        <div class="circular-stat-content">
                            <div class="circular-stat-label">مصرف شده</div>
                            <div class="circular-stat-value"><?php echo round($usedData / 1024 / 1024 / 1024, 2); ?> GB</div>
                        </div>
                    </div>
                    
                    <div class="circular-stat-item">
                        <div class="circular-stat-icon" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                            <i class="fas fa-arrow-down"></i>
                        </div>
                        <div class="circular-stat-content">
                            <div class="circular-stat-label">باقیمانده</div>
                            <div class="circular-stat-value"><?php echo round($remainingData / 1024 / 1024 / 1024, 2); ?> GB</div>
                        </div>
                    </div>
                    
                    <div class="circular-stat-item">
                        <div class="circular-stat-icon" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                            <i class="fas fa-database"></i>
                        </div>
                        <div class="circular-stat-content">
                            <div class="circular-stat-label">کل حجم</div>
                            <div class="circular-stat-value"><?php echo round($totalData / 1024 / 1024 / 1024, 2); ?> GB</div>
                        </div>
                    </div>
                </div>
            </div>
            <?php else: ?>
            <div class="not-activated-message">
                <div class="not-activated-icon">
                    <i class="fas fa-info-circle"></i>
                </div>
                <div>
                    <div class="not-activated-title">eSIM هنوز فعال نشده است</div>
                    <div class="not-activated-desc">پس از فعال‌سازی eSIM، اطلاعات مصرف داده در اینجا نمایش داده می‌شود.</div>
                </div>
            </div>
            
            <!-- پیام مهم درباره به‌روزرسانی داده‌ها -->
            <div class="data-update-notice" style="margin-top: 20px; background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); padding: 20px; border-radius: 12px; color: white; display: flex; align-items: flex-start; gap: 15px;">
                <div style="width: 50px; height: 50px; background: rgba(255,255,255,0.25); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.5rem; flex-shrink: 0;">
                    <i class="fas fa-wifi"></i>
                </div>
                <div>
                    <div style="font-size: 1rem; font-weight: 700; margin-bottom: 8px;">
                        <i class="fas fa-lightbulb"></i> نکته مهم
                    </div>
                    <div style="font-size: 0.9rem; opacity: 0.95; line-height: 1.7;">
                        اطلاعات مصرف و وضعیت eSIM <strong>پس از اولین اتصال به شبکه موبایل</strong> به‌روزرسانی می‌شود.
                        <br>
                        لطفاً eSIM را روی گوشی نصب کنید و در کشور مقصد به شبکه متصل شوید، سپس این صفحه را رفرش کنید.
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- Time Remaining -->
        <div class="detail-card">
            <div class="time-box">
                <div class="time-icon-large">
                    <i class="fas fa-hourglass-half"></i>
                </div>
                <div class="time-content">
                    <div class="time-label">مدت زمان باقیمانده</div>
                    <div class="time-value"><?php echo $daysRemaining; ?> روز</div>
                </div>
            </div>
        </div>
        
        <!-- Network Info -->
        <div class="detail-card">
            <div class="detail-card-title">
                <i class="fas fa-broadcast-tower"></i>
                <span>اطلاعات شبکه</span>
            </div>
            
            <div class="info-list">
                <?php if ($apiData && isset($apiData['esimStatus'])): ?>
                <div class="info-row">
                    <span class="info-label">وضعیت eSIM:</span>
                    <span class="info-value <?php echo $isActive ? 'status-active-text' : 'status-inactive-text'; ?>">
                        <?php 
                        $statusMap = [
                            'GOT_RESOURCE' => '✅ آماده نصب',
                            'RELEASED' => '✅ آماده نصب',
                            'INSTALLED' => '📲 نصب شده',
                            'ACTIVATED' => '✅ فعال شده',
                            'ACTIVE' => '🟢 فعال',
                            'IN_USE' => '🟢 در حال استفاده',
                            'DEPLETED' => '⚠️ حجم تمام شده',
                            'EXPIRED' => '❌ منقضی شده',
                            'CANCEL' => '❌ لغو شده'
                        ];
                        $currentStatus = $apiData['esimStatus'] ?? 'نامشخص';
                        $hasInstallTime = !empty($apiData['installationTime']);
                        
                        // اگر installationTime پر شده، یعنی نصب شده
                        if ($hasInstallTime && in_array($currentStatus, ['GOT_RESOURCE', 'RELEASED'])) {
                            echo '📲 نصب شده';
                        } else {
                            echo $statusMap[$currentStatus] ?? '⚪ ' . htmlspecialchars($currentStatus);
                        }
                        ?>
                    </span>
                </div>
                <?php endif; ?>
                
                <?php if ($network): ?>
                <div class="info-row">
                    <span class="info-label">APN / شبکه:</span>
                    <span class="info-value"><?php echo htmlspecialchars($network); ?></span>
                </div>
                <?php endif; ?>
                
                <?php if ($apiData && isset($apiData['ipExport'])): ?>
                <div class="info-row">
                    <span class="info-label">IP Export:</span>
                    <span class="info-value"><?php echo htmlspecialchars($apiData['ipExport']); ?></span>
                </div>
                <?php endif; ?>
                
                <div class="info-row">
                    <span class="info-label">حجم کل:</span>
                    <span class="info-value"><?php echo htmlspecialchars($esim['data_amount'] ?? 'N/A'); ?></span>
                </div>
                
                <?php if ($apiData && isset($apiData['totalDuration'])): ?>
                <div class="info-row">
                    <span class="info-label">مدت اعتبار:</span>
                    <span class="info-value"><?php echo $apiData['totalDuration']; ?> روز</span>
                </div>
                <?php endif; ?>
                
                <?php if ($activationTime): ?>
                <div class="info-row">
                    <span class="info-label">زمان فعال‌سازی:</span>
                    <span class="info-value"><?php echo date('Y/m/d H:i', strtotime($activationTime)); ?></span>
                </div>
                <?php else: ?>
                <div class="info-row">
                    <span class="info-label">زمان فعال‌سازی:</span>
                    <span class="info-value" style="color: #f59e0b;">هنوز فعال نشده</span>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- QR Code -->
        <?php if (!empty($esim['qr_code'])): ?>
        <div class="detail-card full-width">
            <div class="detail-card-title">
                <i class="fas fa-qrcode"></i>
                <span>QR Code</span>
            </div>
            
            <div class="qr-wrapper">
                <img src="<?php echo htmlspecialchars($esim['qr_code']); ?>" alt="QR Code">
                
                <?php if (!empty($esim['activation_code'])): ?>
                <a href="<?php echo htmlspecialchars($esim['activation_code']); ?>" 
                   class="btn-auto-add"
                   target="_blank">
                    <i class="fas fa-mobile-alt"></i>
                    <span>افزودن خودکار به گوشی</span>
                </a>
                <?php endif; ?>
            </div>
            
            <?php if (!empty($esim['activation_code'])): ?>
            <div class="activation-code-box">
                <label class="activation-label">کد فعال‌سازی:</label>
                <div class="activation-code" id="code-<?php echo $esim['id']; ?>">
                    <?php echo htmlspecialchars($esim['activation_code']); ?>
                </div>
                <button class="btn-copy" onclick="copyCode(<?php echo $esim['id']; ?>, '<?php echo htmlspecialchars($esim['activation_code']); ?>')">
                    <i class="fas fa-copy"></i>
                    کپی کد
                </button>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($esim['iccid'])): ?>
            <div class="activation-code-box">
                <label class="activation-label">ICCID:</label>
                <div class="activation-code">
                    <?php echo htmlspecialchars($esim['iccid']); ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        
        <!-- دکمه افزایش حجم -->
        <?php if (!empty($esim['iccid'])): ?>
        <div class="detail-card full-width topup-button-card" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); border: none; transition: all 0.3s ease;">
            <a href="/buy-more-data.php?iccid=<?php echo urlencode($esim['iccid']); ?>" 
               style="display: flex; align-items: center; justify-content: center; gap: 15px; padding: 30px; text-decoration: none; color: white;">
                <i class="fas fa-plus-circle" style="font-size: 2.5rem;"></i>
                <div style="text-align: right;">
                    <div style="font-size: 1.5rem; font-weight: 900; margin-bottom: 8px;">
                        افزایش حجم و اعتبار
                    </div>
                    <div style="font-size: 1rem; opacity: 0.95;">
                        با خرید بسته‌های شارژ مجدد، حجم و مدت اعتبار به این eSIM اضافه می‌شود
                    </div>
                </div>
            </a>
        </div>
        <?php endif; ?>
        
        <style>
        .topup-button-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(16, 185, 129, 0.4) !important;
        }
        </style>
        
        <!-- آموزش نصب -->
        <div class="detail-card full-width">
            <div class="detail-card-title">
                <i class="fas fa-graduation-cap"></i>
                <span>آموزش نصب eSIM</span>
            </div>
            
            <div class="instructions-content">
                <!-- نکته مهم - بالای آموزش -->
                <div class="note-box" style="background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); border: none; color: white; margin-bottom: 20px;">
                    <i class="fas fa-exclamation-triangle" style="color: white; font-size: 1.5rem;"></i>
                    <div style="color: white;">
                        <strong style="color: white; font-size: 1.1rem;">⚠️ نکته بسیار مهم:</strong>
                        <br>
                        <span style="line-height: 1.8; color: white;">
                            برای نصب و فعال‌سازی eSIM:
                            <br>
                            ✅ حتماً به <strong>اینترنت Wi-Fi</strong> متصل باشید (با VPN فعال)
                            <br>
                            ❌ از <strong>هات‌اسپات (Hotspot)</strong> استفاده نکنید
                        </span>
                    </div>
                </div>
                
                <div class="instruction-section">
                    <h4><i class="fab fa-apple"></i> آیفون (iPhone):</h4>
                    <ol>
                        <li>به <strong>Settings</strong> بروید</li>
                        <li>روی <strong>Cellular</strong> یا <strong>Mobile Data</strong> ضربه بزنید</li>
                        <li><strong>Add Cellular Plan</strong> را انتخاب کنید</li>
                        <li>QR Code بالا را اسکن کنید</li>
                        <li>دستورالعمل‌های روی صفحه را دنبال کنید</li>
                    </ol>
                </div>
                
                <div class="instruction-section">
                    <h4><i class="fab fa-android"></i> اندروید (Android):</h4>
                    <ol>
                        <li>به <strong>Settings</strong> بروید</li>
                        <li><strong>Network & Internet</strong> را انتخاب کنید</li>
                        <li>روی <strong>Mobile Network</strong> ضربه بزنید</li>
                        <li><strong>Add</strong> یا <strong>+</strong> را بزنید</li>
                        <li>QR Code را اسکن کنید</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function copyCode(id, code) {
    navigator.clipboard.writeText(code).then(function() {
        alert('✅ کد با موفقیت کپی شد!');
    }, function() {
        alert('❌ خطا در کپی کردن کد');
    });
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

