<?php
require_once __DIR__ . '/../config/config.php';

class OTPService {
    private $apiKey;
    private $patternCode;
    private $sender;
    private $secret;
    
    public function __construct() {
        $this->apiKey = getSetting('ippanel_api_key', 'yscjWq9-KctIRCU7OC3HBejG4PRa5KyjcHzcFVjPy_g=');
        $this->patternCode = getSetting('ippanel_pattern_code', 'lourive2zquwy5h');
        $this->sender = getSetting('ippanel_sender', '+985000125475');
        $this->secret = getSetting('otp_secret', 'default_secret_key');
    }
    
    /**
     * Generate 4-digit OTP
     */
    private function generateOTP() {
        return str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
    }
    
    /**
     * Hash OTP with HMAC-SHA256
     */
    private function hashOTP($otp) {
        return hash_hmac('sha256', $otp, $this->secret);
    }
    
    /**
     * Check rate limit
     */
    private function checkRateLimit($phone, $action, $ipAddress) {
        $conn = getDBConnection();
        
        // Clean old records
        $conn->query("DELETE FROM otp_rate_limits WHERE created_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)");
        
        // Check if blocked
        $stmt = $conn->prepare("SELECT blocked_until FROM otp_rate_limits WHERE phone = ? AND action = ? AND blocked_until > NOW() LIMIT 1");
        $stmt->bind_param("ss", $phone, $action);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $stmt->close();
            $conn->close();
            return [
                'allowed' => false,
                'message' => 'شما موقتاً مسدود شده‌اید. لطفاً بعداً تلاش کنید.',
                'blocked_until' => $row['blocked_until']
            ];
        }
        $stmt->close();
        
        // Count attempts in last hour
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM otp_rate_limits WHERE phone = ? AND action = ? AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)");
        $stmt->bind_param("ss", $phone, $action);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $attempts = $row['count'];
        $stmt->close();
        
        // Check limits
        $maxAttempts = ($action === 'send') ? 3 : 5;
        
        if ($attempts >= $maxAttempts) {
            // Block for 5 minutes
            $blockedUntil = date('Y-m-d H:i:s', strtotime('+5 minutes'));
            $stmt = $conn->prepare("INSERT INTO otp_rate_limits (phone, ip_address, action, attempts, blocked_until, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
            $stmt->bind_param("sssis", $phone, $ipAddress, $action, $attempts, $blockedUntil);
            $stmt->execute();
            $stmt->close();
            $conn->close();
            
            return [
                'allowed' => false,
                'message' => 'تعداد تلاش‌های شما بیش از حد مجاز است. لطفاً 5 دقیقه صبر کنید.',
                'blocked_until' => $blockedUntil
            ];
        }
        
        // Log attempt
        $stmt = $conn->prepare("INSERT INTO otp_rate_limits (phone, ip_address, action, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->bind_param("sss", $phone, $ipAddress, $action);
        $stmt->execute();
        $stmt->close();
        $conn->close();
        
        return ['allowed' => true, 'remaining' => $maxAttempts - $attempts - 1];
    }
    
    /**
     * Send OTP via IPPanel
     */
    public function sendOTP($phone) {
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        // Check rate limit
        $rateCheck = $this->checkRateLimit($phone, 'send', $ipAddress);
        if (!$rateCheck['allowed']) {
            return [
                'success' => false,
                'message' => $rateCheck['message']
            ];
        }
        
        // Generate OTP
        $otp = $this->generateOTP();
        $otpHash = $this->hashOTP($otp);
        
        // Save to database
        $conn = getDBConnection();
        
        // Mark old OTPs as used
        $stmt = $conn->prepare("UPDATE otps SET used = 1 WHERE phone = ? AND used = 0");
        $stmt->bind_param("s", $phone);
        $stmt->execute();
        $stmt->close();
        
        // Insert new OTP
        $expiresAt = date('Y-m-d H:i:s', strtotime('+5 minutes'));
        $stmt = $conn->prepare("INSERT INTO otps (phone, otp_hash, created_at, expires_at) VALUES (?, ?, NOW(), ?)");
        $stmt->bind_param("sss", $phone, $otpHash, $expiresAt);
        $stmt->execute();
        $stmt->close();
        $conn->close();
        
        // Send via IPPanel
        $smsResult = $this->sendSMS($phone, $otp);
        
        if ($smsResult['success']) {
            return [
                'success' => true,
                'message' => 'کد تایید به شماره ' . $phone . ' ارسال شد',
                'expires_in' => 300 // 5 minutes in seconds
            ];
        } else {
            return [
                'success' => false,
                'message' => 'خطا در ارسال پیامک: ' . $smsResult['message']
            ];
        }
    }
    
    /**
     * Send SMS via IPPanel Pattern API
     */
    private function sendSMS($phone, $otp) {
        // Check if in test mode
        $testMode = getSetting('otp_test_mode', '0');
        
        if ($testMode === '1') {
            // Test mode - don't send real SMS, just log
            error_log("🧪 TEST MODE - OTP for $phone: $otp");
            return ['success' => true, 'message' => 'sent (test mode)'];
        }
        
        $url = 'https://api2.ippanel.com/api/v1/sms/pattern/normal/send';
        
        $data = [
            'code' => $this->patternCode,
            'sender' => $this->sender,
            'recipient' => $phone,
            'variable' => [
                'OTP' => $otp
            ]
        ];
        
        // Log request details
        error_log("📤 IPPanel Request - URL: $url");
        error_log("📤 IPPanel Request - API Key: " . substr($this->apiKey, 0, 20) . "...");
        error_log("📤 IPPanel Request - Pattern: " . $this->patternCode);
        error_log("📤 IPPanel Request - Phone: $phone, OTP: $otp");
        error_log("📤 IPPanel Request - Data: " . json_encode($data, JSON_UNESCAPED_UNICODE));
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'apikey: ' . $this->apiKey,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        // Log response
        error_log("📥 IPPanel Response - HTTP: $httpCode");
        error_log("📥 IPPanel Response - Body: $response");
        if ($curlError) {
            error_log("❌ IPPanel cURL Error: $curlError");
        }
        
        if ($httpCode === 200 || $httpCode === 201) {
            return ['success' => true, 'message' => 'sent'];
        } else {
            // Parse error response
            $errorMsg = 'HTTP ' . $httpCode;
            if ($response) {
                $responseData = json_decode($response, true);
                if (isset($responseData['message'])) {
                    $errorMsg .= ' - ' . $responseData['message'];
                } elseif (isset($responseData['error'])) {
                    $errorMsg .= ' - ' . $responseData['error'];
                }
            }
            
            return [
                'success' => false,
                'message' => $curlError ?: $errorMsg
            ];
        }
    }
    
    /**
     * Verify OTP
     */
    public function verifyOTP($phone, $otp) {
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        // Check rate limit
        $rateCheck = $this->checkRateLimit($phone, 'verify', $ipAddress);
        if (!$rateCheck['allowed']) {
            return [
                'success' => false,
                'message' => $rateCheck['message']
            ];
        }
        
        $conn = getDBConnection();
        
        // Get latest unused OTP
        $stmt = $conn->prepare("SELECT id, otp_hash, expires_at, attempts FROM otps WHERE phone = ? AND used = 0 AND expires_at > NOW() ORDER BY created_at DESC LIMIT 1");
        $stmt->bind_param("s", $phone);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            $stmt->close();
            $conn->close();
            return [
                'success' => false,
                'message' => 'کد تایید منقضی شده یا یافت نشد'
            ];
        }
        
        $row = $result->fetch_assoc();
        $stmt->close();
        
        // Check attempts
        if ($row['attempts'] >= 5) {
            $stmt = $conn->prepare("UPDATE otps SET used = 1 WHERE id = ?");
            $stmt->bind_param("i", $row['id']);
            $stmt->execute();
            $stmt->close();
            $conn->close();
            
            return [
                'success' => false,
                'message' => 'تعداد تلاش‌های شما بیش از حد مجاز است'
            ];
        }
        
        // Increment attempts
        $stmt = $conn->prepare("UPDATE otps SET attempts = attempts + 1 WHERE id = ?");
        $stmt->bind_param("i", $row['id']);
        $stmt->execute();
        $stmt->close();
        
        // Verify OTP
        $otpHash = $this->hashOTP($otp);
        
        if (hash_equals($row['otp_hash'], $otpHash)) {
            // Mark as used
            $stmt = $conn->prepare("UPDATE otps SET used = 1 WHERE id = ?");
            $stmt->bind_param("i", $row['id']);
            $stmt->execute();
            $stmt->close();
            $conn->close();
            
            return [
                'success' => true,
                'verified' => true,
                'message' => 'کد تایید صحیح است'
            ];
        } else {
            $conn->close();
            $remaining = 5 - $row['attempts'] - 1;
            
            return [
                'success' => false,
                'message' => 'کد تایید اشتباه است. ' . $remaining . ' تلاش باقی مانده'
            ];
        }
    }
}
