/**
 * WebView Authentication Handler
 * این اسکریپت برای WebView های اپ موبایل طراحی شده
 * و token رو در هر request اضافه می‌کنه
 */

(function() {
    'use strict';
    
    // تشخیص WebView
    function isWebView() {
        const ua = navigator.userAgent || navigator.vendor || window.opera;
        return (ua.indexOf('wv') > -1) || // Android WebView
               (ua.indexOf('WebView') > -1) ||
               (ua.indexOf('Mobile') > -1 && ua.indexOf('Safari') === -1) || // iOS WebView
               window.isNativeApp === true || // اگر اپ خودش این flag رو ست کرده
               sessionStorage.getItem('force_webview_mode') === 'true'; // Force mode برای تست
    }
    
    // Force WebView Mode (برای تست توسط developer)
    window.forceWebViewMode = function() {
        sessionStorage.setItem('force_webview_mode', 'true');
        console.log('✅ WebView Mode فعال شد (برای تست)');
        window.location.reload();
    };
    
    // خواندن token از localStorage
    function getToken() {
        try {
            return localStorage.getItem('auth_token');
        } catch (e) {
            console.error('❌ خطا در خواندن localStorage:', e);
            return null;
        }
    }
    
    // ذخیره token در localStorage
    function saveToken(token) {
        try {
            localStorage.setItem('auth_token', token);
            console.log('✅ Token در localStorage ذخیره شد');
            return true;
        } catch (e) {
            console.error('❌ خطا در ذخیره localStorage:', e);
            return false;
        }
    }
    
    // اضافه کردن token به URL
    function addTokenToUrl(url) {
        const token = getToken();
        if (!token) return url;
        
        try {
            const urlObj = new URL(url, window.location.origin);
            urlObj.searchParams.set('auth_token', token);
            return urlObj.toString();
        } catch (e) {
            // fallback for relative URLs
            const separator = url.includes('?') ? '&' : '?';
            return url + separator + 'auth_token=' + encodeURIComponent(token);
        }
    }
    
    // Intercept all link clicks
    function interceptLinks() {
        document.addEventListener('click', function(e) {
            const link = e.target.closest('a');
            if (!link) return;
            
            const href = link.getAttribute('href');
            if (!href || href.startsWith('#') || href.startsWith('javascript:') || 
                href.startsWith('mailto:') || href.startsWith('tel:')) {
                return;
            }
            
            // اگر external link است، token اضافه نکن
            if (href.startsWith('http') && !href.includes(window.location.hostname)) {
                return;
            }
            
            const token = getToken();
            if (token && !href.includes('auth_token=')) {
                e.preventDefault();
                const newUrl = addTokenToUrl(href);
                console.log('🔗 Redirect با token:', newUrl.substring(0, 100) + '...');
                window.location.href = newUrl;
            }
        }, true);
    }
    
    // Intercept form submissions
    function interceptForms() {
        document.addEventListener('submit', function(e) {
            const form = e.target;
            if (!form || form.tagName !== 'FORM') return;
            
            const token = getToken();
            if (!token) return;
            
            // اضافه کردن token به عنوان hidden field
            let tokenInput = form.querySelector('input[name="auth_token"]');
            if (!tokenInput) {
                tokenInput = document.createElement('input');
                tokenInput.type = 'hidden';
                tokenInput.name = 'auth_token';
                form.appendChild(tokenInput);
            }
            tokenInput.value = token;
            console.log('📝 Token به form اضافه شد');
        }, true);
    }
    
    // Intercept fetch requests
    const originalFetch = window.fetch;
    window.fetch = function(url, options) {
        const token = getToken();
        if (token) {
            options = options || {};
            options.headers = options.headers || {};
            
            // اضافه کردن token به header
            if (typeof options.headers.set === 'function') {
                options.headers.set('Authorization', 'Bearer ' + token);
            } else {
                options.headers['Authorization'] = 'Bearer ' + token;
            }
            
            console.log('📡 Token به fetch request اضافه شد');
        }
        return originalFetch.apply(this, arguments);
    };
    
    // Intercept XMLHttpRequest
    const originalOpen = XMLHttpRequest.prototype.open;
    const originalSend = XMLHttpRequest.prototype.send;
    
    XMLHttpRequest.prototype.open = function(method, url) {
        this._url = url;
        return originalOpen.apply(this, arguments);
    };
    
    XMLHttpRequest.prototype.send = function() {
        const token = getToken();
        if (token) {
            this.setRequestHeader('Authorization', 'Bearer ' + token);
            console.log('📡 Token به XHR request اضافه شد');
        }
        return originalSend.apply(this, arguments);
    };
    
    // Check token در URL و ذخیره در localStorage
    function checkUrlToken() {
        const urlParams = new URLSearchParams(window.location.search);
        const urlToken = urlParams.get('auth_token');
        
        if (urlToken) {
            console.log('🔍 Token در URL یافت شد:', urlToken.substring(0, 20) + '...');
            saveToken(urlToken);
            
            // حذف token از URL (برای امنیت)
            urlParams.delete('auth_token');
            const newUrl = window.location.pathname + 
                          (urlParams.toString() ? '?' + urlParams.toString() : '') +
                          window.location.hash;
            
            if (window.history && window.history.replaceState) {
                window.history.replaceState({}, document.title, newUrl);
                console.log('🧹 Token از URL حذف شد');
            }
        }
    }
    
    // ست کردن Cookie از localStorage (برای PHP)
    function syncCookie() {
        const token = getToken();
        if (token) {
            try {
                document.cookie = 'auth_token=' + token + '; path=/; max-age=' + (30*24*60*60) + '; SameSite=Lax';
                console.log('🍪 Cookie sync شد');
            } catch (e) {
                console.warn('⚠️ نتونست Cookie ست کنه:', e);
            }
        }
    }
    
    // Initialize
    function init() {
        console.log('🚀 WebView Auth Handler initialized');
        console.log('📱 Is WebView:', isWebView());
        
        // چک token در URL
        checkUrlToken();
        
        // Sync cookie
        syncCookie();
        
        const token = getToken();
        console.log('🔑 Token Status:', token ? 'موجود (' + token.substring(0, 20) + '...)' : 'موجود نیست');
        
        // فقط اگر در WebView هستیم، interceptor ها رو فعال کن
        if (isWebView()) {
            interceptLinks();
            interceptForms();
        }
        
        // Expose API برای اپ native
        window.WebViewAuth = {
            getToken: getToken,
            saveToken: saveToken,
            isLoggedIn: function() {
                return !!getToken();
            },
            logout: function() {
                localStorage.removeItem('auth_token');
                localStorage.removeItem('user_data');
                document.cookie = 'auth_token=; path=/; max-age=0';
                console.log('🚪 Logged out');
            }
        };
    }
    
    // Run on DOMContentLoaded
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();

