<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/api/esim-api.php';

requireLogin();

// Get ICCID from URL  
$iccid = isset($_GET['iccid']) ? trim($_GET['iccid']) : '';

if (empty($iccid)) {
    header('Location: /my-esims.php');
    exit;
}

// Get eSIM from database
$conn = getDBConnection();
$stmt = $conn->prepare("SELECT * FROM orders WHERE iccid = ? AND user_id = ? AND status = 'paid'");
$stmt->bind_param("si", $iccid, $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
$esim = $result->fetch_assoc();
$stmt->close();
$conn->close();

if (!$esim) {
    header('Location: /my-esims.php');
    exit;
}

// Initialize API
$api = new eSIMAPI();

// Check if eSIM was actually created (has ICCID)
if (empty($esim['iccid'])) {
    $apiError = 'این eSIM هنوز فعال نشده است. لطفاً چند دقیقه صبر کنید و دوباره تلاش کنید.';
    $topupPackages = [];
} else {
    // Get TOPUP packages for this eSIM
    $topupPackages = [];
    $apiError = '';

    try {
        // Log the request
        error_log("Requesting topup packages for ICCID: " . $iccid);
        
        $response = $api->getTopupPackages($iccid);
        
        error_log("Topup packages API response: " . json_encode($response));
        
        if ($response && isset($response['success']) && $response['success'] === true) {
            if (isset($response['obj']['packageList']) && is_array($response['obj']['packageList'])) {
                $topupPackages = $response['obj']['packageList'];
                error_log("Found " . count($topupPackages) . " topup packages");
            } else {
                error_log("No packageList in response");
            }
        } else {
            $errorMsg = $response['errorMsg'] ?? 'خطا در دریافت بسته‌های شارژ';
            $apiError = $errorMsg;
            error_log("API Error: " . $errorMsg);
            
            // If ICCID is invalid, try to get packages by country code only
            if (strpos(strtolower($errorMsg), 'iccid') !== false) {
                error_log("ICCID seems invalid, trying with country code only");
                $response2 = $api->getTopupPackages('', $esim['country_code']);
                
                if ($response2 && isset($response2['success']) && $response2['success'] === true) {
                    if (isset($response2['obj']['packageList']) && is_array($response2['obj']['packageList'])) {
                        $topupPackages = $response2['obj']['packageList'];
                        $apiError = ''; // Clear error if we found packages
                        error_log("Found packages by country code: " . count($topupPackages));
                    }
                }
            }
        }
    } catch (Exception $e) {
        $apiError = 'خطا در ارتباط با API: ' . $e->getMessage();
        error_log("Exception in topup packages: " . $e->getMessage());
    }
}

$pageTitle = 'شارژ مجدد eSIM';
include __DIR__ . '/includes/header.php';
?>

<style>
@import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;600;700;800&display=swap');

body, h1, h2, h3, h4, h5, h6, p, span, div, a, button, input, textarea, select, label {
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif !important;
}

.fas, .far, .fab, .fa {
    font-family: "Font Awesome 6 Free", "Font Awesome 6 Brands" !important;
}

.container {
    max-width: 900px;
    margin: 0 auto;
    padding: 30px 20px;
}

.info-box {
    background: linear-gradient(135deg, #CB1919 0%, #FF2929 100%);
    color: white;
    padding: 40px;
    border-radius: 20px;
    text-align: center;
    margin-bottom: 30px;
    box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
}

.info-box h1 {
    font-size: 2.5rem;
    font-weight: 900;
    margin-bottom: 15px;
}

.info-box p {
    font-size: 1.2rem;
    opacity: 0.95;
    line-height: 1.8;
}

.current-esim {
    background: white;
    padding: 30px;
    border-radius: 15px;
    margin-bottom: 30px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.current-esim h2 {
    color: #1e293b;
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 12px;
}

.current-esim h2 i {
    color: #CB1919;
}

.esim-details {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-top: 20px;
}

.detail-item {
    background: #f8fafc;
    padding: 20px;
    border-radius: 12px;
    border: 2px solid #e2e8f0;
}

.detail-item .label {
    color: #64748b;
    font-size: 0.9rem;
    font-weight: 600;
    margin-bottom: 8px;
}

.detail-item .value {
    color: #1e293b;
    font-size: 1.1rem;
    font-weight: 800;
}

.alert-info {
    background: #e0e7ff;
    border: 2px solid #c7d2fe;
    color: #3730a3;
    padding: 25px;
    border-radius: 15px;
    margin-bottom: 30px;
    display: flex;
    align-items: flex-start;
    gap: 15px;
}

.alert-info i {
    font-size: 2rem;
    margin-top: 5px;
}

.alert-info div {
    flex: 1;
}

.alert-info h3 {
    font-size: 1.3rem;
    font-weight: 800;
    margin-bottom: 10px;
}

.alert-info p {
    line-height: 1.8;
    margin: 8px 0;
}

.alert-warning {
    background: #fef3c7;
    border: 2px solid #fbbf24;
    color: #92400e;
    padding: 25px;
    border-radius: 15px;
    margin-bottom: 30px;
}

.alert-warning h3 {
    font-weight: 800;
    margin-bottom: 10px;
}

.options-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 25px;
    margin-bottom: 30px;
}

.option-card {
    background: white;
    border: 3px solid #e2e8f0;
    border-radius: 15px;
    padding: 30px;
    text-align: center;
    transition: all 0.3s ease;
    cursor: pointer;
}

.option-card:hover {
    border-color: #CB1919;
    transform: translateY(-5px);
    box-shadow: 0 10px 30px rgba(102, 126, 234, 0.2);
}

.option-card .icon {
    font-size: 4rem;
    margin-bottom: 20px;
}

.option-card.recommended {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    border-color: #10b981;
}

.option-card.recommended .icon {
    color: white;
}

.option-card h3 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 15px;
}

.option-card p {
    line-height: 1.8;
    margin-bottom: 25px;
}

.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    padding: 15px 35px;
    border: none;
    border-radius: 12px;
    font-size: 1.1rem;
    font-weight: 800;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #CB1919 0%, #FF2929 100%);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
}

.btn-success {
    background: white;
    color: #059669;
    border: 2px solid white;
}

.btn-success:hover {
    background: rgba(255,255,255,0.9);
}

.btn-secondary {
    background: #e2e8f0;
    color: #475569;
}

.btn-secondary:hover {
    background: #cbd5e1;
}

@media (max-width: 768px) {
    body {
        padding-top: 70px !important;
    }
    
    .container {
        padding: 15px 12px;
    }
    
    .info-box {
        padding: 25px 20px;
        border-radius: 16px;
    }
    
    .info-box h1 {
        font-size: 1.4rem;
    }
    
    .info-box p {
        font-size: 1rem;
    }
    
    .current-esim {
        padding: 20px 18px;
    }
    
    .current-esim h2 {
        font-size: 1.2rem;
    }
    
    .esim-details {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .detail-item {
        padding: 16px;
    }
    
    .alert-info, .alert-warning {
        padding: 18px;
        flex-direction: column;
    }
    
    .alert-info i {
        font-size: 1.5rem;
        margin-top: 0;
    }
    
    .alert-info h3 {
        font-size: 1.1rem;
    }
    
    .options-grid {
        grid-template-columns: 1fr;
        gap: 18px;
    }
    
    .option-card {
        padding: 25px 20px;
    }
    
    .option-card .icon {
        font-size: 3rem;
        margin-bottom: 15px;
    }
    
    .option-card h3 {
        font-size: 1.3rem;
    }
    
    .btn {
        padding: 13px 25px;
        font-size: 0.95rem;
        width: 100%;
    }
}

@media (max-width: 480px) {
    .container {
        padding: 12px 10px;
    }
    
    .info-box {
        padding: 20px 16px;
    }
    
    .info-box h1 {
        font-size: 1.2rem;
    }
    
    .info-box p {
        font-size: 0.9rem;
    }
    
    .current-esim {
        padding: 18px 15px;
    }
    
    .current-esim h2 {
        font-size: 1.1rem;
    }
    
    .detail-item {
        padding: 14px;
    }
    
    .detail-item .label {
        font-size: 0.85rem;
    }
    
    .detail-item .value {
        font-size: 1rem;
    }
    
    .alert-info, .alert-warning {
        padding: 16px;
    }
    
    .alert-info h3 {
        font-size: 1rem;
    }
    
    .alert-info p {
        font-size: 0.9rem;
    }
    
    .option-card {
        padding: 20px 16px;
    }
    
    .option-card .icon {
        font-size: 2.5rem;
    }
    
    .option-card h3 {
        font-size: 1.2rem;
    }
    
    .btn {
        padding: 12px 20px;
        font-size: 0.9rem;
    }
}
</style>

<div class="container">
    <div class="info-box">
        <h1>
            <i class="fas fa-redo-alt"></i>
            شارژ مجدد eSIM
        </h1>
        <p>با خرید بسته شارژ، حجم و اعتبار eSIM خود را افزایش دهید</p>
    </div>
    
    <div class="current-esim">
        <h2>
            <i class="fas fa-sim-card"></i>
            eSIM فعلی شما
        </h2>
        
        <div class="esim-details">
            <div class="detail-item">
                <div class="label">کشور</div>
                <div class="value"><?php echo htmlspecialchars($esim['country_name'] ?? 'نامشخص'); ?></div>
            </div>
            <div class="detail-item">
                <div class="label">بسته فعلی</div>
                <div class="value"><?php echo htmlspecialchars($esim['package_name'] ?? 'نامشخص'); ?></div>
            </div>
            <div class="detail-item">
                <div class="label">حجم فعلی</div>
                <div class="value"><?php echo htmlspecialchars($esim['data_amount'] ?? 'نامشخص'); ?></div>
            </div>
            <div class="detail-item">
                <div class="label">ICCID</div>
                <div class="value" style="font-size:0.85rem; word-break: break-all;"><?php echo htmlspecialchars($esim['iccid']); ?></div>
            </div>
        </div>
    </div>
    
    <?php if (!empty($apiError)): ?>
        <div class="alert-warning">
            <h3>
                <i class="fas fa-exclamation-triangle"></i>
                خطا در دریافت بسته‌های شارژ
            </h3>
            <p><?php echo htmlspecialchars($apiError); ?></p>
            <p>لطفاً بعداً مجدداً تلاش کنید یا با پشتیبانی تماس بگیرید.</p>
        </div>
    <?php endif; ?>
    
    <?php if (empty($topupPackages) && empty($apiError)): ?>
        <div class="alert-info">
            <i class="fas fa-info-circle"></i>
            <div>
                <h3>بسته شارژ موجود نیست</h3>
                <p>متأسفانه در حال حاضر بسته شارژ برای این eSIM در دسترس نیست.</p>
                <p><strong>راه حل جایگزین:</strong> می‌توانید یک eSIM جدید برای همین کشور خریداری کنید.</p>
            </div>
        </div>
        
        <div style="text-align: center; margin: 30px 0;">
            <a href="/country.php?code=<?php echo htmlspecialchars($esim['country_code']); ?>" class="btn btn-primary">
                <i class="fas fa-shopping-bag"></i>
                خرید eSIM جدید برای <?php echo htmlspecialchars($esim['country_name']); ?>
            </a>
        </div>
    <?php else: ?>
        <div class="alert-info">
            <i class="fas fa-bolt"></i>
            <div>
                <h3>بسته‌های شارژ موجود</h3>
                <p>با خرید هر کدام از بسته‌های زیر، حجم و اعتبار eSIM شما افزایش می‌یابد.</p>
                <p>⚡ شارژ بلافاصله پس از پرداخت فعال می‌شود</p>
            </div>
        </div>
        
        <div class="options-grid">
            <?php foreach ($topupPackages as $package): 
                $volumeGB = round($package['volume'] / (1024 * 1024 * 1024), 1);
                $priceUSD = $package['price'] / 10000;
                $priceIRR = usdToIrr($priceUSD);
            ?>
                <div class="option-card">
                    <div class="icon" style="color: #10b981;">
                        <i class="fas fa-plus-circle"></i>
                    </div>
                    <h3><?php echo $volumeGB; ?> GB</h3>
                    <p style="color: #666; margin: 10px 0;">
                        <i class="fas fa-calendar-alt"></i>
                        <?php echo $package['duration']; ?> روز اعتبار
                    </p>
                    <p style="font-size: 0.9rem; color: #888; margin-bottom: 20px;">
                        <?php echo htmlspecialchars($package['name'] ?? ''); ?>
                    </p>
                    <div style="font-size: 1.8rem; font-weight: 800; color: #CB1919; margin-bottom: 20px;">
                        <?php echo number_format($priceIRR); ?> تومان
                    </div>
                    <a href="/checkout-topup.php?iccid=<?php echo urlencode($esim['iccid']); ?>&package=<?php echo urlencode($package['packageCode']); ?>" 
                       class="btn btn-primary">
                        <i class="fas fa-shopping-cart"></i>
                        خرید بسته شارژ
                    </a>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
    
    <div style="text-align: center; margin-top: 40px; padding-top: 30px; border-top: 2px solid #e2e8f0;">
        <h3 style="color: #64748b; margin-bottom: 20px;">یا می‌توانید...</h3>
        <div style="display: flex; gap: 15px; justify-content: center; flex-wrap: wrap;">
            <a href="/country.php?code=<?php echo htmlspecialchars($esim['country_code']); ?>" class="btn btn-secondary">
                <i class="fas fa-plus"></i>
                خرید eSIM جدید برای <?php echo htmlspecialchars($esim['country_name']); ?>
            </a>
            <a href="/" class="btn btn-secondary">
                <i class="fas fa-globe"></i>
                انتخاب کشور دیگر
            </a>
            <a href="/my-esims.php" class="btn btn-secondary">
                <i class="fas fa-arrow-right"></i>
                بازگشت به لیست
            </a>
        </div>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>



