<?php
// User Management Functions

// Get user by mobile
function getUserByMobile($mobile) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM users WHERE mobile = ?");
    $stmt->bind_param("s", $mobile);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    return $user;
}

// Get user by ID
function getUserById($userId) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    return $user;
}

// Create new user
function createUser($mobile, $email, $password) {
    $conn = getDBConnection();
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("INSERT INTO users (mobile, email, password) VALUES (?, ?, ?)");
    $stmt->bind_param("sss", $mobile, $email, $hashedPassword);
    $result = $stmt->execute();
    $userId = $conn->insert_id;
    $stmt->close();
    $conn->close();
    
    return $result ? $userId : false;
}

// Login user
function loginUser($mobile, $password) {
    $user = getUserByMobile($mobile);
    
    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['mobile'] = $user['mobile'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['is_admin'] = $user['is_admin'];
        return true;
    }
    
    return false;
}

// Logout user
function logoutUser() {
    session_unset();
    session_destroy();
}

// Order Management Functions

// Create new order
function createOrder($userId, $packageData, $priceUsd) {
    $conn = getDBConnection();
    $priceIrr = usdToIrr($priceUsd);
    
    $stmt = $conn->prepare("INSERT INTO orders (user_id, country_code, country_name, package_name, package_code, data_amount, validity_days, price_usd, price_irr, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
    $stmt->bind_param("isssssidd",
        $userId,
        $packageData['country_code'],
        $packageData['country_name'],
        $packageData['package_name'],
        $packageData['package_code'],
        $packageData['data_amount'],
        $packageData['validity_days'],
        $priceUsd,
        $priceIrr
    );
    
    $result = $stmt->execute();
    $orderId = $conn->insert_id;
    $stmt->close();
    $conn->close();
    
    return $result ? $orderId : false;
}

// Get order by ID
function getOrderById($orderId) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
    $stmt->bind_param("i", $orderId);
    $stmt->execute();
    $result = $stmt->get_result();
    $order = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    return $order;
}

// Update order status
function updateOrderStatus($orderId, $status, $paymentData = []) {
    $conn = getDBConnection();
    
    if (isset($paymentData['authority'])) {
        $stmt = $conn->prepare("UPDATE orders SET status = ?, payment_authority = ? WHERE id = ?");
        $stmt->bind_param("ssi", $status, $paymentData['authority'], $orderId);
    } elseif (isset($paymentData['ref_id'])) {
        $stmt = $conn->prepare("UPDATE orders SET status = ?, payment_ref_id = ?, paid_at = NOW() WHERE id = ?");
        $stmt->bind_param("ssi", $status, $paymentData['ref_id'], $orderId);
    } else {
        $stmt = $conn->prepare("UPDATE orders SET status = ? WHERE id = ?");
        $stmt->bind_param("si", $status, $orderId);
    }
    
    $result = $stmt->execute();
    $stmt->close();
    $conn->close();
    return $result;
}

// Update order with eSIM data
function updateOrderWithESIM($orderId, $esimData) {
    $conn = getDBConnection();
    
    $stmt = $conn->prepare("UPDATE orders SET esim_id = ?, iccid = ?, qr_code = ?, activation_code = ? WHERE id = ?");
    $stmt->bind_param("ssssi",
        $esimData['esim_id'],
        $esimData['iccid'],
        $esimData['qr_code'],
        $esimData['activation_code'],
        $orderId
    );
    
    $result = $stmt->execute();
    $stmt->close();
    $conn->close();
    return $result;
}

// Get user orders
function getUserOrders($userId, $limit = 10) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM orders WHERE user_id = ? ORDER BY created_at DESC LIMIT ?");
    $stmt->bind_param("ii", $userId, $limit);
    $stmt->execute();
    $result = $stmt->get_result();
    $orders = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    $conn->close();
    return $orders;
}

// Get all orders (for admin)
function getAllOrders($limit = 50) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT o.*, u.mobile, u.email FROM orders o JOIN users u ON o.user_id = u.id ORDER BY o.created_at DESC LIMIT ?");
    $stmt->bind_param("i", $limit);
    $stmt->execute();
    $result = $stmt->get_result();
    $orders = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    $conn->close();
    return $orders;
}

