<?php
header('Content-Type: application/json; charset=utf-8');
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../includes/functions.php';

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);
$phone = $input['phone'] ?? '';
$email = $input['email'] ?? '';

// Validate
$phone = convertPersianToEnglish($phone);
if (!validateMobile($phone)) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'شماره موبایل نامعتبر است']);
    exit;
}

if (!validateEmail($email)) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'آدرس ایمیل نامعتبر است']);
    exit;
}

// Check if user already exists
$existingUser = getUserByMobile($phone);
if ($existingUser) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'این شماره موبایل قبلاً ثبت شده است']);
    exit;
}

// Create user (no password needed for OTP)
$conn = getDBConnection();
$stmt = $conn->prepare("INSERT INTO users (mobile, email, created_at) VALUES (?, ?, NOW())");
$stmt->bind_param("ss", $phone, $email);

if ($stmt->execute()) {
    $userId = $stmt->insert_id;
    
    // Login user
    $_SESSION['user_id'] = $userId;
    $_SESSION['mobile'] = $phone;
    $_SESSION['email'] = $email;
    $_SESSION['is_admin'] = 0;
    
    // Regenerate session ID for security
    session_regenerate_id(true);
    
    // Generate auth token for WebView/mobile apps
    $token = bin2hex(random_bytes(32));
    $expiresAt = date('Y-m-d H:i:s', strtotime('+30 days'));
    $stmt = $conn->prepare("INSERT INTO auth_tokens (user_id, token, expires_at) VALUES (?, ?, ?)");
    $stmt->bind_param("iss", $userId, $token, $expiresAt);
    $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    echo json_encode([
        'status' => 'ok',
        'message' => 'ثبت نام موفقیت‌آمیز',
        'token' => $token,
        'user' => [
            'id' => $userId,
            'mobile' => $phone,
            'email' => $email,
            'is_admin' => 0
        ]
    ]);
} else {
    $stmt->close();
    $conn->close();
    
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'خطا در ثبت نام']);
}

