<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/api/esim-api.php';

requireLogin();

$pageTitle = 'eSIM های من';

// Get paid orders (active eSIMs)
$esims = getUserOrders($_SESSION['user_id'], 'paid');

// Initialize API
$api = new eSIMAPI();

include __DIR__ . '/includes/header.php';
?>

<style>
@import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600;700;800&display=swap');

:root {
    --primary: #6366f1;
    --primary-dark: #4f46e5;
    --primary-light: #818cf8;
    --secondary: #ec4899;
    --success: #10b981;
    --warning: #f59e0b;
    --danger: #ef4444;
    --info: #3b82f6;
    --dark: #1e293b;
    --light: #f8fafc;
    --gray-50: #f9fafb;
    --gray-100: #f3f4f6;
    --gray-200: #e5e7eb;
    --gray-300: #d1d5db;
    --gray-400: #9ca3af;
    --gray-500: #6b7280;
    --gray-600: #4b5563;
    --gray-700: #374151;
    --gray-800: #1f2937;
    --gray-900: #111827;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow: 0 1px 3px 0 rgb(0 0 0 / 0.1), 0 1px 2px -1px rgb(0 0 0 / 0.1);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
    --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 8px 10px -6px rgb(0 0 0 / 0.1);
    --shadow-2xl: 0 25px 50px -12px rgb(0 0 0 / 0.25);
    --radius-sm: 8px;
    --radius: 12px;
    --radius-lg: 16px;
    --radius-xl: 20px;
    --radius-2xl: 24px;
    --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: 'Vazirmatn', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    min-height: 100vh;
    padding: 20px;
    direction: rtl;
}

.esim-container {
    max-width: 1400px;
    margin: 0 auto;
}

/* Hero Header */
.hero-header {
    background: rgba(255, 255, 255, 0.1);
    backdrop-filter: blur(20px);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: var(--radius-2xl);
    padding: 40px;
    margin-bottom: 40px;
    position: relative;
    overflow: hidden;
    box-shadow: var(--shadow-2xl);
}

.hero-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: pulse 8s ease-in-out infinite;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 0.5; }
    50% { transform: scale(1.1); opacity: 0.8; }
}

.hero-content {
    position: relative;
    z-index: 1;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 20px;
}

.hero-title {
    display: flex;
    align-items: center;
    gap: 15px;
}

.hero-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
    border-radius: var(--radius-lg);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 28px;
    box-shadow: var(--shadow-lg);
    animation: float 3s ease-in-out infinite;
}

@keyframes float {
    0%, 100% { transform: translateY(0px); }
    50% { transform: translateY(-10px); }
}

.hero-text h1 {
    color: white;
    font-size: 2.5rem;
    font-weight: 800;
    margin-bottom: 5px;
    text-shadow: 0 2px 10px rgba(0,0,0,0.2);
}

.hero-text p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 1rem;
    font-weight: 400;
}

.hero-actions {
    display: flex;
    gap: 15px;
}

.btn-refresh {
    background: rgba(255, 255, 255, 0.2);
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.3);
    color: white;
    padding: 12px 24px;
    border-radius: var(--radius-lg);
    font-weight: 600;
    font-size: 0.95rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 10px;
    transition: var(--transition);
    box-shadow: var(--shadow-md);
}

.btn-refresh:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-refresh i {
    font-size: 1.1rem;
    animation: spin 2s linear infinite paused;
}

.btn-refresh:hover i {
    animation-play-state: running;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

/* Stats Bar */
.stats-bar {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 40px;
}

.stat-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(10px);
    border-radius: var(--radius-xl);
    padding: 25px;
    display: flex;
    align-items: center;
    gap: 20px;
    box-shadow: var(--shadow-lg);
    transition: var(--transition);
    border: 1px solid rgba(255, 255, 255, 0.5);
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-2xl);
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: var(--radius-lg);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    flex-shrink: 0;
}

.stat-icon.active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.stat-icon.expired {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.stat-icon.total {
    background: linear-gradient(135deg, #6366f1 0%, #4f46e5 100%);
    color: white;
}

.stat-info h3 {
    font-size: 2rem;
    font-weight: 800;
    color: var(--gray-900);
    margin-bottom: 5px;
}

.stat-info p {
    font-size: 0.9rem;
    color: var(--gray-600);
    font-weight: 500;
}

/* eSIM Grid */
.esims-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(380px, 1fr));
    gap: 30px;
    margin-bottom: 40px;
}

/* eSIM Card */
.esim-card {
    background: white;
    border-radius: var(--radius-2xl);
    overflow: hidden;
    box-shadow: var(--shadow-xl);
    transition: var(--transition);
    position: relative;
    border: 1px solid var(--gray-200);
}

.esim-card:hover {
    transform: translateY(-8px);
    box-shadow: var(--shadow-2xl);
}

.esim-card-header {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    padding: 25px;
    position: relative;
    overflow: hidden;
}

.esim-card-header::before {
    content: '';
    position: absolute;
    top: -50%;
    left: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
}

.esim-card-header.expired {
    background: linear-gradient(135deg, var(--gray-600) 0%, var(--gray-700) 100%);
}

.card-top {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 20px;
    position: relative;
    z-index: 1;
}

.country-info {
    display: flex;
    align-items: center;
    gap: 15px;
}

.country-flag {
    width: 60px;
    height: 45px;
    border-radius: var(--radius-sm);
    object-fit: cover;
    box-shadow: var(--shadow-md);
    border: 2px solid rgba(255, 255, 255, 0.3);
}

.country-name h3 {
    color: white;
    font-size: 1.4rem;
    font-weight: 700;
    margin-bottom: 5px;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.country-name p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 0.9rem;
    font-weight: 500;
}

.status-badge {
    padding: 8px 16px;
    border-radius: 50px;
    font-size: 0.85rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 6px;
    backdrop-filter: blur(10px);
    box-shadow: var(--shadow-sm);
}

.status-badge.active {
    background: rgba(16, 185, 129, 0.2);
    border: 1px solid rgba(16, 185, 129, 0.3);
    color: #10b981;
}

.status-badge.expired {
    background: rgba(239, 68, 68, 0.2);
    border: 1px solid rgba(239, 68, 68, 0.3);
    color: #ef4444;
}

.status-badge i {
    font-size: 0.7rem;
    animation: blink 2s ease-in-out infinite;
}

@keyframes blink {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.3; }
}

.card-details {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 15px;
    position: relative;
    z-index: 1;
}

.detail-badge {
    background: rgba(255, 255, 255, 0.15);
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 12px;
    border-radius: var(--radius);
    text-align: center;
}

.detail-badge i {
    color: rgba(255, 255, 255, 0.8);
    font-size: 1.2rem;
    margin-bottom: 8px;
    display: block;
}

.detail-badge span {
    color: white;
    font-size: 0.85rem;
    font-weight: 600;
    display: block;
}

/* Card Body */
.esim-card-body {
    padding: 25px;
}

/* Live Data Section */
.live-data-section {
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border-radius: var(--radius-lg);
    padding: 20px;
    margin-bottom: 20px;
    border: 1px solid #bae6fd;
}

.section-title {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 20px;
    color: var(--gray-800);
    font-weight: 700;
    font-size: 1.1rem;
}

.section-title i {
    color: var(--info);
    font-size: 1.2rem;
}

/* Circular Progress */
.usage-circle {
    display: flex;
    justify-content: center;
    align-items: center;
    margin-bottom: 20px;
    position: relative;
}

.circle-container {
    position: relative;
    width: 180px;
    height: 180px;
}

.circle-svg {
    transform: rotate(-90deg);
}

.circle-bg {
    fill: none;
    stroke: #e0e7ff;
    stroke-width: 12;
}

.circle-progress {
    fill: none;
    stroke: url(#gradient);
    stroke-width: 12;
    stroke-linecap: round;
    transition: stroke-dashoffset 1s ease;
}

.circle-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    text-align: center;
}

.circle-percentage {
    font-size: 2.5rem;
    font-weight: 800;
    color: var(--primary);
    display: block;
    line-height: 1;
}

.circle-label {
    font-size: 0.85rem;
    color: var(--gray-600);
    font-weight: 600;
    margin-top: 5px;
    display: block;
}

.usage-stats {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 15px;
    margin-bottom: 20px;
}

.usage-stat {
    text-align: center;
    padding: 15px;
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow-sm);
}

.usage-stat-value {
    font-size: 1.5rem;
    font-weight: 800;
    color: var(--gray-900);
    display: block;
    margin-bottom: 5px;
}

.usage-stat-label {
    font-size: 0.8rem;
    color: var(--gray-600);
    font-weight: 600;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 12px;
}

.info-item {
    background: white;
    padding: 15px;
    border-radius: var(--radius);
    display: flex;
    align-items: center;
    gap: 12px;
    box-shadow: var(--shadow-sm);
    transition: var(--transition);
}

.info-item:hover {
    box-shadow: var(--shadow-md);
    transform: translateX(-3px);
}

.info-icon {
    width: 40px;
    height: 40px;
    border-radius: var(--radius-sm);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    flex-shrink: 0;
}

.info-icon.status-active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.info-icon.status-inactive {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.info-icon.default {
    background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
    color: white;
}

.info-content {
    flex: 1;
}

.info-label {
    font-size: 0.8rem;
    color: var(--gray-600);
    font-weight: 600;
    margin-bottom: 3px;
}

.info-value {
    font-size: 0.95rem;
    color: var(--gray-900);
    font-weight: 700;
}

/* QR Section */
.qr-section {
    background: var(--gray-50);
    border-radius: var(--radius-lg);
    padding: 20px;
    text-align: center;
    margin-bottom: 20px;
    border: 2px dashed var(--gray-300);
}

.qr-code-wrapper {
    display: inline-block;
    padding: 15px;
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow-md);
    margin-bottom: 15px;
}

.qr-code-wrapper img {
    display: block;
    max-width: 200px;
    height: auto;
    border-radius: var(--radius-sm);
}

.qr-hint {
    color: var(--gray-600);
    font-size: 0.9rem;
    font-weight: 500;
    margin-bottom: 15px;
}

/* Buttons */
.btn-auto-add {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    padding: 14px 28px;
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    border: none;
    border-radius: var(--radius-lg);
    font-weight: 700;
    font-size: 1rem;
    cursor: pointer;
    transition: var(--transition);
    box-shadow: var(--shadow-lg);
    text-decoration: none;
    width: 100%;
}

.btn-auto-add:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-xl);
    background: linear-gradient(135deg, var(--primary-dark) 0%, var(--primary) 100%);
}

.btn-auto-add i {
    font-size: 1.2rem;
}

/* Activation Code */
.activation-section {
    margin-bottom: 20px;
}

.activation-label {
    font-size: 0.9rem;
    color: var(--gray-700);
    font-weight: 700;
    margin-bottom: 10px;
    display: block;
}

.activation-code-box {
    display: flex;
    gap: 10px;
}

.activation-code {
    flex: 1;
    background: var(--gray-100);
    border: 2px solid var(--gray-300);
    border-radius: var(--radius);
    padding: 15px;
    font-family: 'Courier New', monospace;
    font-size: 0.85rem;
    color: var(--gray-800);
    word-break: break-all;
    direction: ltr;
    text-align: left;
}

.btn-copy {
    background: linear-gradient(135deg, var(--success) 0%, #059669 100%);
    color: white;
    border: none;
    padding: 15px 20px;
    border-radius: var(--radius);
    cursor: pointer;
    font-weight: 700;
    transition: var(--transition);
    box-shadow: var(--shadow-md);
    white-space: nowrap;
}

.btn-copy:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

.btn-copy i {
    margin-left: 5px;
}

/* ICCID Display */
.iccid-box {
    background: var(--gray-50);
    border: 1px solid var(--gray-200);
    border-radius: var(--radius);
    padding: 15px;
    margin-bottom: 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.iccid-label {
    font-size: 0.85rem;
    color: var(--gray-600);
    font-weight: 700;
}

.iccid-value {
    font-family: 'Courier New', monospace;
    font-size: 0.9rem;
    color: var(--gray-900);
    font-weight: 600;
    direction: ltr;
}

/* Action Buttons */
.action-buttons {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 12px;
}

.btn-action {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: var(--radius);
    font-weight: 600;
    font-size: 0.9rem;
    cursor: pointer;
    transition: var(--transition);
    border: none;
    text-decoration: none;
}

.btn-outline {
    background: white;
    color: var(--primary);
    border: 2px solid var(--primary);
}

.btn-outline:hover {
    background: var(--primary);
    color: white;
}

.btn-primary {
    background: linear-gradient(135deg, var(--info) 0%, #2563eb 100%);
    color: white;
    box-shadow: var(--shadow-md);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
}

/* Instructions */
.instructions-panel {
    background: var(--gray-50);
    border: 1px solid var(--gray-200);
    border-radius: var(--radius-lg);
    padding: 25px;
    margin-top: 20px;
    display: none;
}

.instructions-panel.show {
    display: block;
    animation: slideDown 0.3s ease;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.instructions-title {
    display: flex;
    align-items: center;
    gap: 10px;
    color: var(--gray-900);
    font-weight: 700;
    font-size: 1.1rem;
    margin-bottom: 20px;
}

.instruction-block {
    margin-bottom: 25px;
}

.instruction-block:last-child {
    margin-bottom: 0;
}

.instruction-subtitle {
    display: flex;
    align-items: center;
    gap: 10px;
    color: var(--gray-800);
    font-weight: 700;
    font-size: 1rem;
    margin-bottom: 12px;
}

.instruction-block ol {
    padding-right: 25px;
    margin: 0;
}

.instruction-block li {
    color: var(--gray-700);
    font-weight: 500;
    margin-bottom: 8px;
    line-height: 1.6;
}

.alert {
    padding: 15px 20px;
    border-radius: var(--radius);
    margin-top: 15px;
    display: flex;
    align-items: flex-start;
    gap: 12px;
}

.alert i {
    font-size: 1.2rem;
    margin-top: 2px;
}

.alert-info {
    background: #dbeafe;
    border-right: 4px solid #3b82f6;
    color: #1e40af;
}

.alert-warning {
    background: #fef3c7;
    border-right: 4px solid #f59e0b;
    color: #92400e;
}

/* Empty State */
.empty-state {
    background: white;
    border-radius: var(--radius-2xl);
    padding: 80px 40px;
    text-align: center;
    box-shadow: var(--shadow-xl);
}

.empty-icon {
    width: 120px;
    height: 120px;
    margin: 0 auto 30px;
    background: linear-gradient(135deg, var(--gray-200) 0%, var(--gray-300) 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 4rem;
    color: var(--gray-400);
}

.empty-state h2 {
    color: var(--gray-900);
    font-size: 2rem;
    font-weight: 800;
    margin-bottom: 15px;
}

.empty-state p {
    color: var(--gray-600);
    font-size: 1.1rem;
    margin-bottom: 30px;
}

.btn-large {
    display: inline-flex;
    align-items: center;
    gap: 12px;
    padding: 18px 40px;
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    border: none;
    border-radius: var(--radius-lg);
    font-weight: 700;
    font-size: 1.1rem;
    cursor: pointer;
    transition: var(--transition);
    box-shadow: var(--shadow-xl);
    text-decoration: none;
}

.btn-large:hover {
    transform: translateY(-3px);
    box-shadow: var(--shadow-2xl);
}

/* Responsive */
@media (max-width: 1200px) {
    .esims-grid {
        grid-template-columns: repeat(auto-fill, minmax(340px, 1fr));
    }
}

@media (max-width: 768px) {
    body {
        padding: 15px;
    }
    
    .hero-header {
        padding: 25px;
    }
    
    .hero-content {
        flex-direction: column;
        text-align: center;
    }
    
    .hero-title {
        flex-direction: column;
    }
    
    .hero-text h1 {
        font-size: 1.8rem;
    }
    
    .hero-actions {
        width: 100%;
        flex-direction: column;
    }
    
    .btn-refresh {
        width: 100%;
        justify-content: center;
    }
    
    .stats-bar {
        grid-template-columns: 1fr;
    }
    
    .esims-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .card-details {
        grid-template-columns: 1fr;
        gap: 10px;
    }
    
    .usage-stats {
        grid-template-columns: 1fr;
    }
    
    .info-grid {
        grid-template-columns: 1fr;
    }
    
    .activation-code-box {
        flex-direction: column;
    }
    
    .btn-copy {
        width: 100%;
    }
    
    .action-buttons {
        grid-template-columns: 1fr;
    }
    
    .circle-container {
        width: 150px;
        height: 150px;
    }
    
    .circle-percentage {
        font-size: 2rem;
    }
}

@media (max-width: 480px) {
    .hero-text h1 {
        font-size: 1.5rem;
    }
    
    .stat-card {
        padding: 20px;
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
        font-size: 20px;
    }
    
    .stat-info h3 {
        font-size: 1.5rem;
    }
}
</style>

<div class="esim-container">
    <!-- Hero Header -->
    <div class="hero-header">
        <div class="hero-content">
            <div class="hero-title">
                <div class="hero-icon">
                    <i class="fas fa-sim-card"></i>
                </div>
                <div class="hero-text">
                    <h1>eSIM های من</h1>
                    <p>مدیریت و مشاهده تمام eSIM های خریداری شده</p>
                </div>
            </div>
            <div class="hero-actions">
                <button onclick="location.reload()" class="btn-refresh">
                    <i class="fas fa-sync-alt"></i>
                    <span>بروزرسانی</span>
                </button>
            </div>
        </div>
    </div>

    <?php if (!empty($esims)): ?>
        <?php
        // Calculate stats
        $totalEsims = count($esims);
        $activeEsims = 0;
        $expiredEsims = 0;
        
        foreach ($esims as $esim) {
            if (isset($esim['validity_days']) && isset($esim['created_at'])) {
                $createdTime = strtotime($esim['created_at']);
                $expiryTime = $createdTime + ($esim['validity_days'] * 24 * 60 * 60);
                if ($expiryTime < time()) {
                    $expiredEsims++;
                } else {
                    $activeEsims++;
                }
            }
        }
        ?>
        
        <!-- Stats Bar -->
        <div class="stats-bar">
            <div class="stat-card">
                <div class="stat-icon total">
                    <i class="fas fa-sim-card"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $totalEsims; ?></h3>
                    <p>کل eSIM ها</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon active">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $activeEsims; ?></h3>
                    <p>فعال</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon expired">
                    <i class="fas fa-times-circle"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $expiredEsims; ?></h3>
                    <p>منقضی شده</p>
                </div>
            </div>
        </div>

        <!-- eSIMs Grid -->
        <div class="esims-grid">
            <?php foreach ($esims as $esim): 
                // Get real-time data from API
                $apiData = null;
                $isExpired = false;
                $isActive = false;
                $remainingData = null;
                $totalData = null;
                $usedData = null;
                $expiryDate = null;
                $activationTime = null;
                $network = null;
                
                // Query API for eSIM details if ICCID exists
                if (!empty($esim['iccid'])) {
                    try {
                        $apiResponse = $api->queryProfile($esim['iccid']);
                        
                        // Check for both old format (code) and new format (success/errorCode)
                        $isSuccess = false;
                        if (isset($apiResponse['code']) && $apiResponse['code'] == 0) {
                            $isSuccess = true;
                        } elseif (isset($apiResponse['success']) && $apiResponse['success'] === true) {
                            $isSuccess = true;
                        } elseif (isset($apiResponse['errorCode']) && $apiResponse['errorCode'] === '000000') {
                            $isSuccess = true;
                        }
                        
                        if ($apiResponse && $isSuccess) {
                            // Check if we have esimList in the response
                            $esimList = $apiResponse['obj']['esimList'] ?? ($apiResponse['obj'] ?? null);
                            
                            if (is_array($esimList) && !empty($esimList)) {
                                $apiData = is_array($esimList[0]) ? $esimList[0] : $esimList;
                                
                                // Get status
                                $isActive = isset($apiData['status']) && $apiData['status'] == 'ACTIVE';
                                
                                // Get expiry date
                                if (isset($apiData['expireTime'])) {
                                    $expiryDate = $apiData['expireTime'];
                                    $expiryTimestamp = strtotime($expiryDate);
                                    $isExpired = $expiryTimestamp < time();
                                }
                                
                                // Get activation time
                                if (isset($apiData['activationTime'])) {
                                    $activationTime = $apiData['activationTime'];
                                }
                                
                                // Get network
                                if (isset($apiData['network'])) {
                                    $network = $apiData['network'];
                                }
                                
                                // Get data usage
                                if (isset($apiData['dataUsage'])) {
                                    $totalData = $apiData['dataUsage']['total'] ?? 0;
                                    $usedData = $apiData['dataUsage']['used'] ?? 0;
                                    $remainingData = $totalData - $usedData;
                                }
                            }
                        }
                    } catch (Exception $e) {
                        // Silently fail and use local data
                    }
                }
                
                // Fallback to local data if API fails
                if (!$apiData && isset($esim['validity_days']) && isset($esim['created_at'])) {
                    $createdTime = strtotime($esim['created_at']);
                    $expiryTime = $createdTime + ($esim['validity_days'] * 24 * 60 * 60);
                    $isExpired = $expiryTime < time();
                    $expiryDate = date('Y-m-d H:i:s', $expiryTime);
                }
                
                $cardHeaderClass = $isExpired ? 'esim-card-header expired' : 'esim-card-header';
                $statusBadgeClass = $isExpired ? 'status-badge expired' : 'status-badge active';
                $statusText = $isExpired ? 'منقضی شده' : 'فعال';
                $statusIcon = $isExpired ? 'fa-times-circle' : 'fa-check-circle';
                
                // Calculate usage percentage
                $usagePercentage = 0;
                if ($totalData > 0 && $usedData !== null) {
                    $usagePercentage = round(($usedData / $totalData) * 100, 1);
                }
                
                // Calculate circle progress
                $circumference = 2 * pi() * 80; // radius = 80
                $dashOffset = $circumference - ($usagePercentage / 100 * $circumference);
            ?>
            <div class="esim-card">
                <!-- Card Header -->
                <div class="<?php echo $cardHeaderClass; ?>">
                    <div class="card-top">
                        <div class="country-info">
                            <?php if (!empty($esim['country_code'])): ?>
                            <img src="https://flagcdn.com/w80/<?php echo strtolower($esim['country_code']); ?>.png" 
                                 alt="<?php echo htmlspecialchars($esim['country_name'] ?? ''); ?>"
                                 class="country-flag"
                                 onerror="this.style.display='none'">
                            <?php endif; ?>
                            <div class="country-name">
                                <h3><?php echo htmlspecialchars($esim['package_name'] ?? 'eSIM'); ?></h3>
                                <p><?php echo htmlspecialchars($esim['country_name'] ?? ''); ?></p>
                            </div>
                        </div>
                        <div class="<?php echo $statusBadgeClass; ?>">
                            <i class="fas <?php echo $statusIcon; ?>"></i>
                            <span><?php echo $statusText; ?></span>
                        </div>
                    </div>
                    
                    <div class="card-details">
                        <div class="detail-badge">
                            <i class="fas fa-database"></i>
                            <span><?php echo htmlspecialchars($esim['data_amount'] ?? 'N/A'); ?></span>
                        </div>
                        <div class="detail-badge">
                            <i class="fas fa-calendar"></i>
                            <span><?php echo ($esim['validity_days'] ?? 0); ?> روز</span>
                        </div>
                        <div class="detail-badge">
                            <i class="fas fa-clock"></i>
                            <span><?php echo date('Y/m/d', strtotime($esim['created_at'])); ?></span>
                        </div>
                    </div>
                </div>

                <!-- Card Body -->
                <div class="esim-card-body">
                    <?php if ($apiData && $totalData !== null && $usedData !== null): ?>
                    <!-- Live Data Section -->
                    <div class="live-data-section">
                        <div class="section-title">
                            <i class="fas fa-chart-line"></i>
                            <span>اطلاعات زنده</span>
                        </div>
                        
                        <!-- Circular Progress -->
                        <div class="usage-circle">
                            <div class="circle-container">
                                <svg class="circle-svg" width="180" height="180">
                                    <defs>
                                        <linearGradient id="gradient-<?php echo $esim['id']; ?>" x1="0%" y1="0%" x2="100%" y2="100%">
                                            <stop offset="0%" style="stop-color:#6366f1;stop-opacity:1" />
                                            <stop offset="100%" style="stop-color:#ec4899;stop-opacity:1" />
                                        </linearGradient>
                                    </defs>
                                    <circle class="circle-bg" cx="90" cy="90" r="80"></circle>
                                    <circle class="circle-progress" cx="90" cy="90" r="80"
                                            stroke="url(#gradient-<?php echo $esim['id']; ?>)"
                                            stroke-dasharray="<?php echo $circumference; ?>"
                                            stroke-dashoffset="<?php echo $dashOffset; ?>"></circle>
                                </svg>
                                <div class="circle-text">
                                    <span class="circle-percentage"><?php echo $usagePercentage; ?>%</span>
                                    <span class="circle-label">مصرف شده</span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Usage Stats -->
                        <div class="usage-stats">
                            <div class="usage-stat">
                                <span class="usage-stat-value"><?php echo round($totalData / 1024 / 1024 / 1024, 2); ?></span>
                                <span class="usage-stat-label">کل (GB)</span>
                            </div>
                            <div class="usage-stat">
                                <span class="usage-stat-value"><?php echo round($usedData / 1024 / 1024 / 1024, 2); ?></span>
                                <span class="usage-stat-label">مصرف (GB)</span>
                            </div>
                            <div class="usage-stat">
                                <span class="usage-stat-value"><?php echo round($remainingData / 1024 / 1024 / 1024, 2); ?></span>
                                <span class="usage-stat-label">باقیمانده (GB)</span>
                            </div>
                        </div>
                        
                        <!-- Info Grid -->
                        <div class="info-grid">
                            <div class="info-item">
                                <div class="info-icon <?php echo $isActive ? 'status-active' : 'status-inactive'; ?>">
                                    <i class="fas fa-signal"></i>
                                </div>
                                <div class="info-content">
                                    <div class="info-label">وضعیت</div>
                                    <div class="info-value"><?php echo $isActive ? 'فعال' : 'غیرفعال'; ?></div>
                                </div>
                            </div>
                            
                            <?php if ($expiryDate): ?>
                            <div class="info-item">
                                <div class="info-icon default">
                                    <i class="fas fa-calendar-times"></i>
                                </div>
                                <div class="info-content">
                                    <div class="info-label">انقضا</div>
                                    <div class="info-value"><?php echo date('Y/m/d', strtotime($expiryDate)); ?></div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if ($activationTime): ?>
                            <div class="info-item">
                                <div class="info-icon default">
                                    <i class="fas fa-power-off"></i>
                                </div>
                                <div class="info-content">
                                    <div class="info-label">فعال‌سازی</div>
                                    <div class="info-value"><?php echo date('Y/m/d', strtotime($activationTime)); ?></div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if ($network): ?>
                            <div class="info-item">
                                <div class="info-icon default">
                                    <i class="fas fa-network-wired"></i>
                                </div>
                                <div class="info-content">
                                    <div class="info-label">شبکه</div>
                                    <div class="info-value"><?php echo htmlspecialchars($network); ?></div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (!empty($esim['qr_code'])): ?>
                    <!-- QR Section -->
                    <div class="qr-section">
                        <div class="qr-code-wrapper">
                            <img src="<?php echo htmlspecialchars($esim['qr_code']); ?>" alt="QR Code">
                        </div>
                        <p class="qr-hint">QR Code را با دوربین گوشی اسکن کنید</p>
                        
                        <?php if (!empty($esim['activation_code'])): ?>
                        <a href="<?php echo htmlspecialchars($esim['activation_code']); ?>" 
                           class="btn-auto-add"
                           target="_blank">
                            <i class="fas fa-mobile-alt"></i>
                            <span>افزودن خودکار به گوشی</span>
                        </a>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>

                    <?php if (!empty($esim['activation_code'])): ?>
                    <!-- Activation Code -->
                    <div class="activation-section">
                        <label class="activation-label">کد فعال‌سازی:</label>
                        <div class="activation-code-box">
                            <div class="activation-code" id="code-<?php echo $esim['id']; ?>">
                                <?php echo htmlspecialchars($esim['activation_code']); ?>
                            </div>
                            <button class="btn-copy" onclick="copyCode(<?php echo $esim['id']; ?>, '<?php echo htmlspecialchars($esim['activation_code']); ?>')">
                                <i class="fas fa-copy"></i>
                                <span>کپی</span>
                            </button>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (!empty($esim['iccid'])): ?>
                    <!-- ICCID -->
                    <div class="iccid-box">
                        <span class="iccid-label">ICCID:</span>
                        <span class="iccid-value"><?php echo htmlspecialchars($esim['iccid']); ?></span>
                    </div>
                    <?php endif; ?>

                    <!-- Action Buttons -->
                    <div class="action-buttons">
                        <button class="btn-action btn-outline" onclick="toggleInstructions(<?php echo $esim['id']; ?>)">
                            <i class="fas fa-question-circle"></i>
                            <span>آموزش نصب</span>
                        </button>
                        <?php if (!empty($esim['qr_code'])): ?>
                        <a href="<?php echo htmlspecialchars($esim['qr_code']); ?>" download class="btn-action btn-primary">
                            <i class="fas fa-download"></i>
                            <span>دانلود QR</span>
                        </a>
                        <?php endif; ?>
                    </div>

                    <!-- Instructions Panel -->
                    <div id="instructions-<?php echo $esim['id']; ?>" class="instructions-panel">
                        <div class="instructions-title">
                            <i class="fas fa-mobile-alt"></i>
                            <span>نحوه نصب eSIM</span>
                        </div>
                        
                        <div class="instruction-block">
                            <div class="instruction-subtitle">
                                <i class="fab fa-apple"></i>
                                <span>iOS (iPhone):</span>
                            </div>
                            <ol>
                                <li>به <strong>Settings</strong> بروید</li>
                                <li><strong>Cellular</strong> یا <strong>Mobile Data</strong> را انتخاب کنید</li>
                                <li>روی <strong>Add Cellular Plan</strong> کلیک کنید</li>
                                <li>QR Code را اسکن کنید یا کد را دستی وارد کنید</li>
                                <li>دستورالعمل‌های روی صفحه را دنبال کنید</li>
                            </ol>
                        </div>

                        <div class="instruction-block">
                            <div class="instruction-subtitle">
                                <i class="fab fa-android"></i>
                                <span>Android:</span>
                            </div>
                            <ol>
                                <li>به <strong>Settings</strong> بروید</li>
                                <li><strong>Network & Internet</strong> را انتخاب کنید</li>
                                <li>روی <strong>Mobile Network</strong> کلیک کنید</li>
                                <li><strong>Add Carrier</strong> یا <strong>Download SIM</strong> را بزنید</li>
                                <li>QR Code را اسکن کنید</li>
                            </ol>
                        </div>

                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i>
                            <div>
                                <strong>نکته مهم:</strong> دستگاه شما باید از eSIM پشتیبانی کند. برای اطمینان، از سازنده گوشی خود استعلام کنید.
                            </div>
                        </div>

                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            <div>
                                <strong>هشدار:</strong> هر eSIM فقط یک بار قابل نصب است. قبل از حذف، مطمئن شوید که نیاز به آن ندارید.
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php else: ?>
        <!-- Empty State -->
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-sim-card"></i>
            </div>
            <h2>هنوز eSIM ای خریداری نکرده‌اید</h2>
            <p>برای خرید eSIM، کشور مورد نظر خود را انتخاب کنید.</p>
            <a href="/" class="btn-large">
                <i class="fas fa-shopping-cart"></i>
                <span>خرید eSIM</span>
            </a>
        </div>
    <?php endif; ?>
</div>

<script>
function copyCode(id, code) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(code).then(() => {
            const btn = event.target.closest('.btn-copy');
            const originalHTML = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-check"></i><span>کپی شد!</span>';
            btn.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
            
            setTimeout(() => {
                btn.innerHTML = originalHTML;
                btn.style.background = '';
            }, 2000);
        }).catch(err => {
            alert('خطا در کپی کردن');
        });
    } else {
        const textarea = document.createElement('textarea');
        textarea.value = code;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        document.body.appendChild(textarea);
        textarea.select();
        try {
            document.execCommand('copy');
            alert('کد کپی شد!');
        } catch (err) {
            alert('خطا در کپی کردن');
        }
        document.body.removeChild(textarea);
    }
}

function toggleInstructions(id) {
    const panel = document.getElementById('instructions-' + id);
    if (panel) {
        panel.classList.toggle('show');
    }
}

// Add smooth scroll behavior
document.documentElement.style.scrollBehavior = 'smooth';
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

