/**
 * Token-based Authentication Helper for WebView/Mobile Apps
 * این اسکریپت به طور خودکار token را به تمام لینک‌ها و فرم‌ها اضافه می‌کند
 * سازگار با مرورگر معمولی و WebView
 */

(function() {
    'use strict';
    
    console.log('🔐 Auth Token Helper loaded');
    
    // Get token from localStorage
    function getAuthToken() {
        try {
            return localStorage.getItem('auth_token');
        } catch (e) {
            console.warn('⚠️ نمی‌تونه localStorage بخونه:', e);
            return null;
        }
    }
    
    // Check if token exists in URL and save to localStorage
    function checkUrlToken() {
        const urlParams = new URLSearchParams(window.location.search);
        const urlToken = urlParams.get('auth_token') || urlParams.get('token');
        
        if (urlToken) {
            console.log('🔍 Token در URL یافت شد');
            try {
                localStorage.setItem('auth_token', urlToken);
                console.log('✅ Token در localStorage ذخیره شد');
                
                // حذف token از URL برای امنیت
                urlParams.delete('auth_token');
                urlParams.delete('token');
                const newUrl = window.location.pathname + 
                              (urlParams.toString() ? '?' + urlParams.toString() : '') +
                              window.location.hash;
                
                if (window.history && window.history.replaceState) {
                    window.history.replaceState({}, document.title, newUrl);
                }
            } catch (e) {
                console.error('❌ خطا در ذخیره token:', e);
            }
        }
    }
    
    // Sync token to Cookie (برای PHP)
    function syncTokenToCookie() {
        const token = getAuthToken();
        if (token) {
            try {
                document.cookie = 'auth_token=' + token + '; path=/; max-age=' + (30*24*60*60) + '; SameSite=Lax';
                console.log('🍪 Token به Cookie sync شد');
            } catch (e) {
                console.warn('⚠️ نتونست Cookie ست کنه');
            }
        }
    }
    
    // Add token to URL
    function addTokenToUrl(url) {
        const token = getAuthToken();
        if (!token) return url;
        
        try {
            const urlObj = new URL(url, window.location.origin);
            // استفاده از auth_token برای سازگاری با PHP
            urlObj.searchParams.set('auth_token', token);
            return urlObj.toString();
        } catch (e) {
            // If URL parsing fails, use simple concatenation
            const separator = url.includes('?') ? '&' : '?';
            return url + separator + 'auth_token=' + encodeURIComponent(token);
        }
    }
    
    // Add token to all internal links
    function addTokenToLinks() {
        const token = getAuthToken();
        if (!token) return;
        
        const links = document.querySelectorAll('a[href]');
        links.forEach(function(link) {
            const href = link.getAttribute('href');
            
            // Skip external links, anchors, and javascript links
            if (!href || href.startsWith('#') || href.startsWith('javascript:') || 
                href.startsWith('http://') || href.startsWith('https://')) {
                return;
            }
            
            // Add token to internal links
            link.href = addTokenToUrl(href);
        });
    }
    
    // Add token to all forms
    function addTokenToForms() {
        const token = getAuthToken();
        if (!token) return;
        
        const forms = document.querySelectorAll('form');
        forms.forEach(function(form) {
            // Skip if form already has token input
            if (form.querySelector('input[name="auth_token"]') || form.querySelector('input[name="token"]')) {
                return;
            }
            
            // Skip forms that go to external URLs
            const action = form.getAttribute('action');
            if (action && action.startsWith('http') && !action.includes(window.location.hostname)) {
                return;
            }
            
            // Add hidden token input
            const tokenInput = document.createElement('input');
            tokenInput.type = 'hidden';
            tokenInput.name = 'auth_token';
            tokenInput.value = token;
            form.appendChild(tokenInput);
        });
    }
    
    // Add token to fetch requests
    const originalFetch = window.fetch;
    window.fetch = function() {
        const token = getAuthToken();
        
        if (token && arguments.length > 0) {
            // Add Authorization header if not already present
            if (arguments.length === 1) {
                arguments[1] = {
                    headers: {
                        'Authorization': 'Bearer ' + token
                    }
                };
            } else if (arguments[1] && !arguments[1].headers) {
                arguments[1].headers = {
                    'Authorization': 'Bearer ' + token
                };
            } else if (arguments[1] && arguments[1].headers && !arguments[1].headers.Authorization) {
                arguments[1].headers.Authorization = 'Bearer ' + token;
            }
        }
        
        return originalFetch.apply(this, arguments);
    };
    
    // Expose API برای استفاده در کد دیگر
    window.AuthToken = {
        get: getAuthToken,
        addToUrl: addTokenToUrl,
        syncCookie: syncTokenToCookie,
        isLoggedIn: function() {
            return !!getAuthToken();
        },
        logout: function() {
            try {
                localStorage.removeItem('auth_token');
                localStorage.removeItem('user_data');
                document.cookie = 'auth_token=; path=/; max-age=0';
                console.log('🚪 Logged out');
            } catch (e) {
                console.error('❌ خطا در logout:', e);
            }
        }
    };
    
    // Initialize on page load
    function init() {
        console.log('🚀 Initializing Auth Token Helper...');
        
        // 1. چک کردن token در URL
        checkUrlToken();
        
        // 2. Sync کردن به Cookie
        syncTokenToCookie();
        
        // 3. اضافه کردن به لینک‌ها و فرم‌ها
        addTokenToLinks();
        addTokenToForms();
        
        const token = getAuthToken();
        if (token) {
            console.log('✅ Token موجود است:', token.substring(0, 20) + '...');
        } else {
            console.log('ℹ️ Token موجود نیست');
        }
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
    // Re-add token when new content is added (for AJAX/dynamic content)
    if (window.MutationObserver) {
        const observer = new MutationObserver(function() {
            addTokenToLinks();
            addTokenToForms();
        });
        
        if (document.body) {
            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
        }
    }
    
})();

