<?php
require_once __DIR__ . '/config/config.php';

// If already logged in, redirect to dashboard
if (isLoggedIn()) {
    header('Location: /dashboard.php');
    exit;
}

$pageTitle = 'ورود / ثبت نام';
include __DIR__ . '/includes/header.php';
?>

<style>
@import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;600;700;800&display=swap');

body {
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif !important;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    min-height: 100vh;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 15px;
}

.login-container {
    background: white;
    border-radius: 20px;
    padding: 40px 35px;
    max-width: 500px;
    width: 100%;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: slideUp 0.5s ease;
}

@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.login-header {
    text-align: center;
    margin-bottom: 35px;
}

.login-header .logo {
    width: 70px;
    height: 70px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 18px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 18px;
    box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
}

.login-header .logo i {
    font-size: 2.2rem;
    color: white;
}

.login-header h1 {
    color: #1e293b;
    font-size: 1.8rem;
    font-weight: 800;
    margin-bottom: 8px;
}

.login-header p {
    color: #64748b;
    font-size: 1rem;
    font-weight: 600;
}

.alert {
    padding: 15px 20px;
    border-radius: 12px;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 12px;
    font-weight: 600;
}

.alert-error {
    background: #fee;
    color: #c00;
    border: 2px solid #fcc;
}

.alert-success {
    background: #efe;
    color: #060;
    border: 2px solid #cfc;
}

.alert i {
    font-size: 1.3rem;
}

.form-group {
    margin-bottom: 22px;
}

.form-group label {
    display: block;
    font-weight: 700;
    color: #334155;
    margin-bottom: 10px;
    font-size: 0.95rem;
}

.form-group input {
    width: 100%;
    padding: 14px;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 1.05rem;
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif;
    transition: all 0.3s ease;
    direction: ltr;
    text-align: center;
    font-weight: 700;
    letter-spacing: 1px;
    box-sizing: border-box;
}

.form-group input:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1);
}

.form-group small {
    display: block;
    margin-top: 8px;
    color: #64748b;
    font-size: 0.95rem;
    text-align: center;
}

.btn {
    width: 100%;
    padding: 16px;
    border: none;
    border-radius: 12px;
    font-size: 1.05rem;
    font-weight: 800;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif;
    box-sizing: border-box;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
}

.btn-secondary {
    background: #e2e8f0;
    color: #475569;
    margin-top: 12px;
}

.btn-secondary:hover {
    background: #cbd5e1;
}

.btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none !important;
}

.step-indicator {
    background: #e0e7ff;
    padding: 15px 20px;
    border-radius: 12px;
    margin-bottom: 25px;
    text-align: center;
    font-weight: 700;
    color: #3730a3;
    font-size: 1.05rem;
}

.otp-timer {
    text-align: center;
    margin: 20px 0;
    font-weight: 700;
    color: #667eea;
    font-size: 1.2rem;
}

.otp-timer i {
    margin-left: 8px;
}

.resend-link {
    text-align: center;
    margin-top: 20px;
}

.resend-link button {
    background: none;
    border: none;
    color: #667eea;
    font-weight: 700;
    cursor: pointer;
    text-decoration: underline;
    font-size: 1.05rem;
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif;
    transition: color 0.3s;
}

.resend-link button:hover {
    color: #764ba2;
}

.resend-link button:disabled {
    color: #94a3b8;
    cursor: not-allowed;
    text-decoration: none;
}

.back-home {
    text-align: center;
    margin-top: 30px;
}

.back-home a {
    color: #667eea;
    font-weight: 700;
    text-decoration: none;
    font-size: 1.05rem;
    transition: color 0.3s;
}

.back-home a:hover {
    color: #764ba2;
}

.back-home a i {
    margin-left: 8px;
}

@media (max-width: 768px) {
    body {
        padding: 10px;
    }
    
    .login-container {
        padding: 30px 20px;
        border-radius: 15px;
    }
    
    .login-header {
        margin-bottom: 25px;
    }
    
    .login-header .logo {
        width: 60px;
        height: 60px;
        border-radius: 15px;
        margin-bottom: 15px;
    }
    
    .login-header .logo i {
        font-size: 1.8rem;
    }
    
    .login-header h1 {
        font-size: 1.4rem;
        margin-bottom: 6px;
    }
    
    .login-header p {
        font-size: 0.9rem;
    }
    
    .alert {
        padding: 12px 15px;
        gap: 10px;
        font-size: 0.9rem;
    }
    
    .alert i {
        font-size: 1.1rem;
    }
    
    .form-group {
        margin-bottom: 18px;
    }
    
    .form-group label {
        font-size: 0.9rem;
        margin-bottom: 8px;
    }
    
    .form-group input {
        padding: 13px;
        font-size: 0.95rem;
        border-radius: 10px;
    }
    
    .form-group small {
        font-size: 0.85rem;
        margin-top: 6px;
    }
    
    .btn {
        padding: 14px;
        font-size: 0.95rem;
        gap: 8px;
        border-radius: 10px;
    }
    
    .btn-secondary {
        margin-top: 10px;
        padding: 12px;
    }
    
    .step-indicator {
        padding: 12px 15px;
        font-size: 0.9rem;
        margin-bottom: 20px;
        border-radius: 10px;
    }
    
    .otp-timer {
        font-size: 1.05rem;
        margin: 15px 0;
    }
    
    .resend-link {
        margin-top: 15px;
    }
    
    .resend-link button {
        font-size: 0.9rem;
    }
    
    .back-home {
        margin-top: 25px;
    }
    
    .back-home a {
        font-size: 0.9rem;
    }
}

@media (max-width: 480px) {
    body {
        padding: 8px;
    }
    
    .login-container {
        padding: 25px 18px;
        border-radius: 12px;
    }
    
    .login-header .logo {
        width: 55px;
        height: 55px;
    }
    
    .login-header .logo i {
        font-size: 1.6rem;
    }
    
    .login-header h1 {
        font-size: 1.3rem;
    }
    
    .login-header p {
        font-size: 0.85rem;
    }
    
    .form-group input {
        padding: 12px;
        font-size: 0.9rem;
    }
    
    .btn {
        padding: 13px;
        font-size: 0.9rem;
    }
}

@media (max-width: 360px) {
    .login-container {
        padding: 20px 15px;
    }
    
    .login-header h1 {
        font-size: 1.2rem;
    }
    
    .form-group input {
        padding: 11px;
        font-size: 0.85rem;
    }
    
    .btn {
        padding: 12px;
        font-size: 0.85rem;
    }
}
</style>

<div class="login-container">
    <div class="login-header">
        <div class="logo">
            <i class="fas fa-sim-card"></i>
        </div>
        <h1>خوش آمدید</h1>
        <p>ورود یا ثبت نام با شماره موبایل</p>
    </div>
    
    <!-- Step 1: Mobile -->
    <div id="step-mobile">
        <div id="mobile-error" class="alert alert-error" style="display: none;"></div>
        <div id="mobile-success" class="alert alert-success" style="display: none;"></div>
        
        <form id="mobileForm">
            <div class="form-group">
                <label><i class="fas fa-mobile-alt"></i> شماره موبایل</label>
                <input type="text" id="mobile" name="mobile" required placeholder="09123456789" maxlength="11" autofocus>
                <small>می‌توانید از اعداد فارسی استفاده کنید</small>
            </div>
            <button type="submit" class="btn btn-primary" id="sendOtpBtn">
                <i class="fas fa-paper-plane"></i>
                ارسال کد تایید
            </button>
        </form>
    </div>
    
    <!-- Step 2: OTP Verification -->
    <div id="step-otp" style="display: none;">
        <div class="step-indicator">
            <i class="fas fa-mobile-alt"></i>
            شماره موبایل: <span id="display-mobile"></span>
        </div>
        
        <div id="otp-error" class="alert alert-error" style="display: none;"></div>
        <div id="otp-success" class="alert alert-success" style="display: none;"></div>
        
        <form id="otpForm">
            <div class="form-group">
                <label><i class="fas fa-shield-alt"></i> کد تایید 4 رقمی</label>
                <input type="text" id="otp" name="otp" required placeholder="1234" maxlength="4" pattern="[0-9]{4}" autofocus>
                <small>کد تایید به شماره موبایل شما ارسال شد</small>
            </div>
            
            <div class="otp-timer" id="otpTimer">
                <i class="fas fa-clock"></i>
                <span id="timerDisplay">05:00</span>
            </div>
            
            <button type="submit" class="btn btn-primary" id="verifyOtpBtn">
                <i class="fas fa-check-circle"></i>
                تایید و ورود
            </button>
            
            <div class="resend-link">
                <button type="button" id="resendBtn" onclick="resendOTP()" disabled>
                    ارسال مجدد کد تایید
                </button>
            </div>
            
            <button type="button" class="btn btn-secondary" onclick="backToMobile()">
                <i class="fas fa-arrow-right"></i>
                تغییر شماره موبایل
            </button>
        </form>
    </div>
    
    <!-- Step 3: Register (if new user) -->
    <div id="step-register" style="display: none;">
        <div class="step-indicator">
            <i class="fas fa-user-plus"></i>
            شماره موبایل: <span id="register-mobile"></span>
        </div>
        
        <div id="register-error" class="alert alert-error" style="display: none;"></div>
        
        <form id="registerForm">
            <div class="form-group">
                <label><i class="fas fa-envelope"></i> آدرس ایمیل</label>
                <input type="email" id="email" name="email" required placeholder="email@example.com" autofocus>
                <small>eSIM به این ایمیل ارسال می‌شود</small>
            </div>
            
            <button type="submit" class="btn btn-primary" id="registerBtn">
                <i class="fas fa-user-plus"></i>
                تکمیل ثبت نام
            </button>
            
            <button type="button" class="btn btn-secondary" onclick="backToMobile()">
                <i class="fas fa-arrow-right"></i>
                بازگشت
            </button>
        </form>
    </div>
    
    <div class="back-home">
        <a href="/">
            <i class="fas fa-home"></i>
            بازگشت به صفحه اصلی
        </a>
    </div>
</div>

<script>
// Convert Persian/Arabic numbers to English
function convertPersianToEnglish(str) {
    const persianNumbers = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    const arabicNumbers = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];
    const englishNumbers = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    
    for (let i = 0; i < 10; i++) {
        str = str.replace(new RegExp(persianNumbers[i], 'g'), englishNumbers[i]);
        str = str.replace(new RegExp(arabicNumbers[i], 'g'), englishNumbers[i]);
    }
    return str;
}

// Apply Persian number conversion to inputs
document.addEventListener('DOMContentLoaded', function() {
    const inputs = document.querySelectorAll('input[type="text"]');
    inputs.forEach(function(input) {
        input.addEventListener('input', function(e) {
            const cursorPos = this.selectionStart;
            const oldValue = this.value;
            const newValue = convertPersianToEnglish(oldValue);
            if (oldValue !== newValue) {
                this.value = newValue;
                this.setSelectionRange(cursorPos, cursorPos);
            }
        });
    });
});

let currentPhone = '';
let otpTimer = null;
let timeLeft = 300; // 5 minutes

function backToMobile() {
    document.getElementById('step-mobile').style.display = 'block';
    document.getElementById('step-otp').style.display = 'none';
    document.getElementById('step-register').style.display = 'none';
    clearTimer();
}

function showError(elementId, message) {
    const errorDiv = document.getElementById(elementId);
    errorDiv.innerHTML = '<i class="fas fa-exclamation-circle"></i> ' + message;
    errorDiv.style.display = 'flex';
}

function hideError(elementId) {
    document.getElementById(elementId).style.display = 'none';
}

function showSuccess(elementId, message) {
    const successDiv = document.getElementById(elementId);
    successDiv.innerHTML = '<i class="fas fa-check-circle"></i> ' + message;
    successDiv.style.display = 'flex';
}

function clearTimer() {
    if (otpTimer) {
        clearInterval(otpTimer);
        otpTimer = null;
    }
}

function startTimer() {
    timeLeft = 300; // Reset to 5 minutes
    document.getElementById('resendBtn').disabled = true;
    
    otpTimer = setInterval(function() {
        timeLeft--;
        
        const minutes = Math.floor(timeLeft / 60);
        const seconds = timeLeft % 60;
        document.getElementById('timerDisplay').textContent = 
            String(minutes).padStart(2, '0') + ':' + String(seconds).padStart(2, '0');
        
        if (timeLeft <= 0) {
            clearTimer();
            document.getElementById('resendBtn').disabled = false;
            document.getElementById('timerDisplay').textContent = 'منقضی شده';
        }
    }, 1000);
}

// Step 1: Send OTP
document.getElementById('mobileForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const mobile = convertPersianToEnglish(document.getElementById('mobile').value);
    const btn = document.getElementById('sendOtpBtn');
    
    hideError('mobile-error');
    hideError('mobile-success');
    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال ارسال...';
    
    try {
        const response = await fetch('/api/send-otp.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ phone: mobile })
        });
        
        const data = await response.json();
        
        if (data.status === 'ok') {
            currentPhone = mobile;
            document.getElementById('display-mobile').textContent = mobile;
            showSuccess('mobile-success', 'کد تایید ارسال شد');
            setTimeout(() => {
                document.getElementById('step-mobile').style.display = 'none';
                document.getElementById('step-otp').style.display = 'block';
                document.getElementById('otp').focus();
                startTimer();
            }, 1000);
        } else {
            showError('mobile-error', data.message);
        }
    } catch (error) {
        showError('mobile-error', 'خطا در ارتباط با سرور');
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<i class="fas fa-paper-plane"></i> ارسال کد تایید';
    }
});

// Step 2: Verify OTP
document.getElementById('otpForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const otp = convertPersianToEnglish(document.getElementById('otp').value);
    const btn = document.getElementById('verifyOtpBtn');
    
    hideError('otp-error');
    hideError('otp-success');
    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال بررسی...';
    
    try {
        const response = await fetch('/api/verify-otp.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ phone: currentPhone, otp: otp })
        });
        
        const data = await response.json();
        
        if (data.status === 'ok' && data.verified) {
            if (data.new_user) {
                // Show registration form
                document.getElementById('register-mobile').textContent = currentPhone;
                document.getElementById('step-otp').style.display = 'none';
                document.getElementById('step-register').style.display = 'block';
                document.getElementById('email').focus();
                clearTimer();
            } else {
                // User logged in, redirect to dashboard
                showSuccess('otp-success', 'ورود موفقیت‌آمیز! در حال انتقال...');
                setTimeout(function() {
                    window.location.href = '/dashboard.php';
                }, 1000);
            }
        } else {
            showError('otp-error', data.message);
        }
    } catch (error) {
        showError('otp-error', 'خطا در ارتباط با سرور');
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<i class="fas fa-check-circle"></i> تایید و ورود';
    }
});

// Step 3: Register
document.getElementById('registerForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const email = document.getElementById('email').value;
    const btn = document.getElementById('registerBtn');
    
    hideError('register-error');
    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال ثبت نام...';
    
    try {
        const response = await fetch('/api/register-user.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ phone: currentPhone, email: email })
        });
        
        const data = await response.json();
        
        if (data.status === 'ok') {
            // Registration successful, redirect to dashboard
            window.location.href = '/dashboard.php';
        } else {
            showError('register-error', data.message);
        }
    } catch (error) {
        showError('register-error', 'خطا در ارتباط با سرور');
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<i class="fas fa-user-plus"></i> تکمیل ثبت نام';
    }
});

// Resend OTP
async function resendOTP() {
    const btn = document.getElementById('resendBtn');
    btn.disabled = true;
    btn.textContent = 'در حال ارسال...';
    
    hideError('otp-error');
    hideError('otp-success');
    
    try {
        const response = await fetch('/api/send-otp.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ phone: currentPhone })
        });
        
        const data = await response.json();
        
        if (data.status === 'ok') {
            showSuccess('otp-success', 'کد جدید ارسال شد');
            setTimeout(() => hideError('otp-success'), 3000);
            startTimer();
        } else {
            showError('otp-error', data.message);
        }
    } catch (error) {
        showError('otp-error', 'خطا در ارتباط با سرور');
    } finally {
        btn.textContent = 'ارسال مجدد کد تایید';
    }
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
